#include "ABF.h"
#include "CredAcceptance.h"
#include "SkeptAcceptance.h"
#include "SingleExtension.h"
#include "EnumerateExtensions.h"
#include "Propagator.h"
#include "Util.h"

#include <iostream>
#include <fstream>
#include <sstream>
#include <algorithm>
#include <getopt.h>

using namespace std;

static int version_flag = 0;
static int usage_flag = 0;
static int formats_flag = 0;
static int problems_flag = 0;

task string_to_task(string problem)
{
	string tmp = problem.substr(0, problem.find("-"));
	if (tmp == "DC") return DC;
	if (tmp == "DS") return DS;
	if (tmp == "SE") return SE;
	if (tmp == "EE") return EE;
	return UNKNOWN_TASK;
}

semantics string_to_sem(string problem)
{
	problem.erase(0, problem.find("-") + 1);
	string tmp = problem.substr(0, problem.find("-"));
	if (tmp == "AD")  return AD;
	if (tmp == "CO")  return CO;
	if (tmp == "PR")  return PR;
	if (tmp == "ST")  return ST;
	if (tmp == "SST") return SST;
	if (tmp == "STG") return STG;
	if (tmp == "ID")  return ID;
	return UNKNOWN_SEM;
}

void print_usage(string solver_name)
{
	cout << "Usage: " << solver_name << " -p <task> -f <file> -fo <format> [-a <query>]\n\n";
	cout << "  <task>      computational problem; for a list of available problems use option --problems\n";
	cout << "  <file>      input assumption-based argumentation framework (ABA)\n";
	cout << "  <format>    file format for input ABA; for a list of available formats use option --formats\n";
	cout << "  <query>     query argument\n\n";
	cout << "Options:\n";
	cout << "  --help      Displays this help message.\n";
	cout << "  --version   Prints version and author information.\n";
	cout << "  --formats   Prints available file formats.\n";
	cout << "  --problems  Prints available computational tasks.\n";
}

void print_version(string solver_name)
{
	cout << solver_name << "-ACYC (ICCMA'25 version)\n";
}

void print_formats()
{
	cout << "[i23,apx]\n";
}

void print_problems()
{
	vector<string> tasks = {"DC","DS","SE","EE"};
	vector<string> sems = {"AD","CO","PR","ST","SST","STG","ID"};
	cout << "[";
	for (uint32_t i = 0; i < tasks.size(); i++) {
		for (uint32_t j = 0; j < sems.size(); j++) {
			string problem = tasks[i] + "-" + sems[j];
			cout << problem;
			if (problem != "EE-ID") cout << ",";
		}
	}
	cout << "]\n";
}

int main(int argc, char ** argv)
{
	ios_base::sync_with_stdio(false);
	cin.tie(NULL);

	if (argc == 1) {
		print_version(argv[0]);
		return 0;
	}

	const struct option longopts[] =
	{
		{"help", no_argument, &usage_flag, 1},
		{"version", no_argument, &version_flag, 1},
		{"formats", no_argument, &formats_flag, 1},
		{"problems", no_argument, &problems_flag, 1},
		{"p", required_argument, 0, 'p'},
		{"f", required_argument, 0, 'f'},
		{"fo", required_argument, 0, 'o'},
		{"a", required_argument, 0, 'a'},
		{0, 0, 0, 0}
	};

	int option_index = 0;
	int opt = 0;
	string task, file, fileformat, query_str;

	while ((opt = getopt_long_only(argc, argv, "", longopts, &option_index)) != -1) {
		switch (opt) {
			case 0:
				break;
			case 'p':
				task = optarg;
				break;
			case 'f':
				file = optarg;
				break;
			case 'o':
				fileformat = optarg;
				break;
			case 'a':
				query_str = optarg;
				break;
			default:
				return 1;
		}
	}

	if (version_flag) {
		print_version(argv[0]);
		return 0;
	}

	if (usage_flag) {
		print_usage(argv[0]);
		return 0;
	}

	if (formats_flag) {
		print_formats();
		return 0;
	}

	if (problems_flag) {
		print_problems();
		return 0;
	}

	if (task.empty()) {
		cerr << argv[0] << ": Task must be specified via -p flag\n";
		return 1;
	}

	if (file.empty()) {
		cerr << argv[0] << ": Input file must be specified via -f flag\n";
		return 1;
	}

	if (fileformat.empty()) {
		fileformat = "i23";
		//cout << "c Defaulting to ICCMA'23 file format\n";
	}

	ifstream input;
	input.open(file);

	if (!input.good()) {
		cerr << argv[0] << ": Cannot open input file\n";
		return 1;
	}

	ABF af = ABF();
	unordered_map<uint32_t,vector<vector<uint32_t>>> rules;

	if (fileformat == "i23") {
		string line;
		af.symbols_as_strings = false;
		while (!input.eof()) {
			getline(input, line);
			istringstream iss(line);
			char key = '#'; iss >> key;
			uint32_t n_symbols, assum, cont, tmp;
			vector<uint32_t> body;
			uint32_t head;
			if (key == '#') continue;
			switch (key) {
			case 'p':
				iss >> n_symbols;
				break;
			case 'a':
				iss >> assum;
				af.add_assumption(assum);
				break;
			case 'c':
				iss >> assum >> cont;
				af.add_contrary(assum, cont);
				break;
			case 'r':
				iss >> head;
				while (iss >> tmp)
					body.push_back(tmp);
				sort(body.begin(), body.end());
				rules[head].push_back(body);
				break;
			default:
				cerr << "c WARNING: Cannot parse line `" << line << "'\n";
			}
		}
	/*} else if (fileformat == "apx") {
		string line, rule, arg, cont;
		af.symbols_as_strings = true;
		while (!input.eof()) {
			getline(input, line);
			line.erase(remove_if(line.begin(), line.end(), ::isspace), line.end());
			if (line.length() == 0 || line[0] == '/' || line[0] == '%') continue;
			if (line.length() < 6) cerr << "Warning: Cannot parse line: " << line << "\n";
			if (line.substr(0,4) == "rule") {
				if (line[4] == '(' && line.find(')') != string::npos) {
					rule = line.substr(5,line.find(')')-5);
					rules.push_back(rule);
				} else {
					cerr << "Warning: Cannot parse line: " << line << "\n";
				}
			} else if (line.substr(0,5) == "assum") {
				if (line[5] == '(' && line.find(')') != string::npos) {
					arg = line.substr(6,line.find(')')-6);
					af.add_assumption(arg);
				} else {
					cerr << "Warning: Cannot parse line: " << line << "\n";
				}
			} else if (line.substr(0,4) == "cont") {
				if (line[4] == '(' && line.find(',') != string::npos && line.find(')') != string::npos) {
					arg = line.substr(5,line.find(',')-5);
					cont = line.substr(line.find(',')+1,line.find(')')-line.find(',')-1);
					af.add_contrary(arg, cont);
				} else {
					cerr << "Warning: Cannot parse line: " << line << "\n";
				}
			} else {
				cerr << "Warning: Cannot parse line: " << line << "\n";
			}
		}*/
	} else {
		cerr << argv[0] << ": Unsupported file format\n";
		return 1;
	}

	input.close();

	for (auto &[head, bodies] : rules) {
		sort(bodies.begin(), bodies.end());
		bodies.erase(std::unique(bodies.begin(), bodies.end()), bodies.end());
		sort(bodies.begin(), bodies.end(),
			[](const vector<uint32_t> & a, const vector<uint32_t> & b){ return a.size() < b.size(); });
		for (uint32_t i = 0; i < bodies.size(); i++) {
			bool flag = false;
			for (uint32_t j = 0; j < i; j++) {
				if (bodies[i].size() > bodies[j].size() && includes(bodies[i].begin(), bodies[i].end(), bodies[j].begin(), bodies[j].end())) {
					flag = true;
					break;
				}
			}
			if (flag) continue;
			vector<uint32_t> rule = { head };
			rule.insert(rule.end(), bodies[i].begin(), bodies[i].end());
			af.add_rule(rule);
		}
	}

	af.sem = string_to_sem(task);

	af.initialize_vars();
	uint32_t query;
	if (string_to_task(task) == DC || string_to_task(task) == DS) {
		if (query_str.empty()) {
			return 1;
		}
		if (af.symbols_as_strings) {
			query = af.symbol_to_int[query_str];
		} else {
			query = stoi(query_str);
			if (query < 1 || query > af.symbols)
				return 1;
			query--;
		}
	}

	SAT_Solver * solver = new SAT_Solver();
	//solver->set("verbose", 3);
	//solver->set("log", 1);
	//solver->set("quiet", 1);
	solver->set("inprocessing", 0);
	//solver->set("phase", 1);
	solver->set("rephase", 0);
	//solver->set("ilb", 0);
	//solver->set("ilbassumptions", 0);
	//solver->set("walk", 0);
	//solver->set("lucky", 0);

	vector<int> edge_vars;
	if (af.sem == ST || af.sem == STG) {
		for (const auto & [source, target] : af.derivation_edges) {
			edge_vars.push_back(af.derived_from_in_edge_var[make_pair(source, target)]);
		}
	} else {
		for (const auto & [source, target] : af.derivation_edges) {
			edge_vars.push_back(af.derived_from_undec_edge_var[make_pair(source, target)]);
		}
	}

	Graph graph = Graph(edge_vars, af.derivation_edges);

	AcyclicityPropagator * propagator = new AcyclicityPropagator(af, graph);
	solver->connect_external_propagator(propagator);
	solver->connect_fixed_listener(propagator);

	for (int var : edge_vars) {
		solver->add_observed_var(var);
	}
	for (auto [_, arc] : graph.var_to_arc) {
		int a = arc.first; int b = arc.second;
		if (a > b) continue;
		if (graph.arcs.contains(make_pair(b,a))) {
			solver->add(-graph.arc_to_var[graph.key(a,b)]);
			solver->add(-graph.arc_to_var[graph.key(b,a)]);
			solver->add(0);
		}
	}

	switch (string_to_task(task)) {
		case DC:
		{
			bool cred_accepted = false;
			switch (string_to_sem(task)) {
				case AD:
					cred_accepted = CredAcceptance::admissible(af, query, solver);
					break;
				case CO:
					cred_accepted = CredAcceptance::complete(af, query, solver);
					break;
				case PR:
					cred_accepted = CredAcceptance::preferred(af, query, solver);
					break;
				case ST:
					cred_accepted = CredAcceptance::stable(af, query, solver);
					break;
				case SST:
					cred_accepted = CredAcceptance::semi_stable(af, query, solver);
					break;
				case STG:
					cred_accepted = CredAcceptance::stage(af, query, solver);
					break;
				case ID:
					cred_accepted = CredAcceptance::ideal(af, query, solver);
					break;
				default:
					cerr << argv[0] << ": Unsupported semantics\n";
					return 1;
			}
#if !defined(OUTPUT)
			cout << (cred_accepted ? "YES" : "NO") << endl;
#endif
			break;
		}
		case DS:
		{
			bool skept_accepted = false;
			switch (string_to_sem(task)) {
				case AD:
					skept_accepted = SkeptAcceptance::admissible(af, query, solver);
					break;
				case CO:
					skept_accepted = SkeptAcceptance::complete(af, query, solver);
					break;
				case PR:
					skept_accepted = SkeptAcceptance::preferred(af, query, solver);
					break;
				case ST:
					skept_accepted = SkeptAcceptance::stable(af, query, solver);
					break;
				case SST:
					skept_accepted = SkeptAcceptance::semi_stable(af, query, solver);
					break;
				case STG:
					skept_accepted = SkeptAcceptance::stage(af, query, solver);
					break;
				case ID:
					skept_accepted = CredAcceptance::ideal(af, query, solver);
					break;
				default:
					cerr << argv[0] << ": Unsupported semantics\n";
					return 1;
			}
#if !defined(OUTPUT)
			cout << (skept_accepted ? "YES" : "NO") << endl;
#endif
			break;
		}
		case SE:
		{
			switch (string_to_sem(task)) {
				case AD:
					SingleExtension::admissible(af, solver);
					break;
				case CO:
					SingleExtension::complete(af, solver);
					break;
				case PR:
					SingleExtension::preferred(af, solver);
					break;
				case ST:
					SingleExtension::stable(af, solver);
					break;
				case SST:
					SingleExtension::semi_stable(af, solver);
					break;
				case STG:
					SingleExtension::stage(af, solver);
					break;
				case ID:
					SingleExtension::ideal(af, solver);
					break;
				default:
					cerr << argv[0] << ": Unsupported semantics\n";
					return 1;
			}
			break;
		}
		case EE:
		{
			switch (string_to_sem(task)) {
				case AD:
					EnumerateExtensions::admissible(af, solver);
					break;
				case CO:
					EnumerateExtensions::complete(af, solver);
					break;
				case PR:
					EnumerateExtensions::preferred(af, solver);
					break;
				case ST:
					EnumerateExtensions::stable(af, solver);
					break;
				case SST:
					EnumerateExtensions::semi_stable(af, solver);
					break;
				case STG:
					EnumerateExtensions::stage(af, solver);
					break;
				case ID:
					SingleExtension::ideal(af, solver);
					break;
				default:
					cerr << argv[0] << ": Unsupported semantics\n";
					return 1;
			}
			break;
		}
		default:
			cerr << argv[0] << ": Unsupported problem\n";
			return 1;
	}

#if !defined(OUTPUT)
	//solver->statistics();
	//solver->resources();
#endif

	delete propagator;
	delete solver;


	return 0;
}
