#!/usr/bin/env python3

import time
start_perf_counter = time.perf_counter()
import argparse
import sys
import os
import csv
from natsort import natsorted
from utils.Af import AF
from utils.PreferredExtensionFinder import PreferredExtensionFinder

problems = ['DC-CO', 'DC-GR', 'SE-GR', 'DC-PR']



def create_witness(extension):
    return 'w ' + ' '.join(natsorted(extension))


def save_analytics_data(file_path, analytics_dict, run_name):
    # Extract the base filename without path and extension
    base_filename = os.path.basename(file_path)

    # Get the directory containing the input file
    base_dir = os.path.dirname(file_path)

    # Create analytics directory if it doesn't exist
    analytics_dir = os.path.join(base_dir, 'analytics')
    os.makedirs(analytics_dir, exist_ok=True)

    # Create run_name directory if it doesn't exist
    run_dir = os.path.join(analytics_dir, run_name)
    os.makedirs(run_dir, exist_ok=True)

    # Create the CSV file path
    csv_file = os.path.join(run_dir, f"{base_filename}.csv")

    # Check if file exists to determine if we need to write headers
    file_exists = os.path.isfile(csv_file)

    # Open file in append mode
    with open(csv_file, 'a', newline='') as f:
        writer = csv.DictWriter(f, fieldnames=analytics_dict.keys())

        # Write headers if file is new
        if not file_exists:
            writer.writeheader()

        # Write the analytics data
        writer.writerow(analytics_dict)



start_parsing = time.perf_counter()
if len(sys.argv) == 1:
    print('SMART 1.0')
    print(
        'Sandra Hoffmann <sandra.hoffmann@fernuni-hagen.de>\n'
        'Isabelle Kuhlmann <isabelle.kuhlmann@fernuni-hagen.de>\n'
        'Matthias Thimm <matthias.thimm@fernuni-hagen.de>'
    )
    exit(0)

# parser for problems and formats flags
parser = argparse.ArgumentParser(add_help=False)
parser.add_argument('--problems', action='store_true', help='Prints all problems this solver supports')
parser.add_argument('--formats', action='store_true', help='Prints all formats this solver supports')
args, remaining_args = parser.parse_known_args()

if args.problems:
    problems = problems
    print(problems)
    exit(0)

if args.formats:
    print('i23', 'apx')
    exit(0)

full_parser = argparse.ArgumentParser(
    description='SMART Argumentation Solver',
    usage='Python3 smart.py -p <task> -f <file> [-a <query>]'
)
full_parser.add_argument('--problems', action='store_true', help='Prints all problems this solver supports')
full_parser.add_argument('--formats', action='store_true', help='Prints all formats this solver supports')
full_parser.add_argument('-p', type=str, metavar='<task>', help='The problem to solve', required=True)
full_parser.add_argument('-f', type=str, metavar='<file>', help='The path to the AF', required=True)
full_parser.add_argument('-a', type=str, metavar='<query>', help='The query argument')

args = full_parser.parse_args(remaining_args)
save_analytics = False
run_name = "Smart_CoAdm_8_epoch_47"

if save_analytics:
    analytics_dict = {
        'argument': None,
        'result': None,
        'prediction': None,
        'heuristic': None,
        'backtracks': 0,
        'runtime_parsing': 0,
        'runtime_optimize_checks': 0,
        'runtime_prediction': 0,
        'runtime_heuristic': 0,
        'runtime_solving': 0,
        'perf_counter_time': 0
    }

problem = args.p
query = args.a
if save_analytics:
    analytics_dict['argument'] = query

af = AF(args.f)
if save_analytics:
    analytics_dict['runtime_parsing'] = time.perf_counter() - start_parsing

if problem == 'DC-GR':
    if query in af.grounded_ext:
        print('YES')
        if save_analytics:
            analytics_dict['result'] = 'YES'
            analytics_dict['heuristic'] = 'grounded'
        print(create_witness(af.grounded_ext))
    else:
        print('NO')
        if save_analytics:
            analytics_dict['result'] = 'NO'
            analytics_dict['heuristic'] = 'grounded'

if problem == 'SE-GR':
    formatted_extension = ', '.join(natsorted(af.grounded_ext))
    print("[" + formatted_extension + "]")


if problem in {'DC-PR', 'DC-CO'}:
    start_optimize_checks = time.perf_counter()
    arg_is_grounded = False

    if query in af.self_attackers:
        print('NO')
        if save_analytics:
            analytics_dict['result'] = 'NO'
            analytics_dict['heuristic'] = 'self_attacker'
            analytics_dict['perf_counter_time'] = (time.perf_counter() - start_perf_counter)
            save_analytics_data(args.f, analytics_dict, run_name)
        exit(0)

    if query in af.grounded_ext:
        arg_is_grounded = True
    else:
        query_attackers = af.arguments[query]['successors']
        query_attacked = af.arguments[query]['predecessors']

        neighbors = query_attackers + query_attacked

        query_conflicts = set(neighbors) & set(af.grounded_ext)
        if query_conflicts:
            print('NO')
            if save_analytics:
                analytics_dict['result'] = 'NO'
                analytics_dict['heuristic'] = 'grounded'
                analytics_dict['perf_counter_time'] = (time.perf_counter() - start_perf_counter)
                save_analytics_data(args.f, analytics_dict, run_name)
            exit(0)
    if save_analytics:
        analytics_dict['runtime_optimize_checks'] = (time.perf_counter() - start_optimize_checks)

    start_solving_perf = time.perf_counter()
    finder = PreferredExtensionFinder(af)
    extension = finder.get_preferred_extension(query, analytics_dict=None)
    time_to_solve = (time.perf_counter() - start_solving_perf)
    if save_analytics:
        analytics_dict['runtime_solving'] = time_to_solve - analytics_dict['runtime_prediction']- analytics_dict['runtime_heuristic']
    if not extension:
        print('NO')
        if save_analytics:
         analytics_dict['result'] = 'NO'
    else:
        print('YES')
        print(create_witness(extension))
        if save_analytics:
         analytics_dict['result'] = 'YES'




if save_analytics:
    analytics_dict['perf_counter_time'] = (time.perf_counter() - start_perf_counter)
# Save analytics at the end if enabled
# if save_analytics:
#     save_analytics_data(args.f, analytics_dict, run_name)
