import re
from collections import deque

import networkx as nx
from natsort import natsorted
from utils.Labelling import Labelling


class AF:
    def __init__(self, file):
        self.di_graph = nx.DiGraph()
        self.filename = file
        self.self_attackers = []
        self.arguments={}
        if file.endswith('i23'):
            self.read_i23(file)
        elif file.endswith('apx'):
            self.read_apx(file)
        self.grounded_ext = self.get_grounded_ext()

    def intersection(self, lst1, lst2):
        # Use of hybrid method to calculate intersection
        temp = set(lst2)
        lst3 = [value for value in lst1 if value in temp]
        return lst3

    def read_i23(self, file):
        with open(file, 'r') as f:
            first_line = f.readline().strip()
            parts = first_line.split()

            if parts[0] == 'p':
                total_nodes = int(parts[2])
            else:
                raise ValueError("Unexpected File Format")

            # Initialize the dictionary for each argument
            for i in range(total_nodes):
                argument = str(i + 1)
                self.di_graph.add_node(argument)
                self.arguments[argument] = {
                    'successors': [],
                    'predecessors': [],
                }

            for line in f:
                line = line.strip()
                if not (line.startswith('#') or line.startswith('p')):
                    attack = line.split(" ")
                    if attack[0] == attack[1]:
                        # Self-attacking argument found
                        self.self_attackers.append(attack[0])
                    self.arguments[attack[0]]['successors'].append(attack[1])
                    self.arguments[attack[1]]['predecessors'].append(attack[0])
                    self.di_graph.add_edge(attack[0], attack[1])


    def read_apx(self, file):
        with open(file, 'r') as f:
            for line in f:
                line = line.strip()
                if line.startswith('arg('):
                    #get argument name
                    match = re.search(r'\((.*?)\)', line)
                    argument = match.group(1)
                    self.di_graph.add_node(argument)
                    self.arguments[argument] = {
                        'successors': [],
                        'predecessors': [],
                    }
                if line.startswith('att('):
                    #get attack
                    match = re.search(r'\((.*?)\)', line)
                    attack = match.group(1).split(",")
                    if attack[0] == attack[1]:
                        # Self-attacking argument found
                        self.self_attackers.append(attack[0])
                    self.arguments[attack[0]]['successors'].append(attack[1])
                    self.arguments[attack[1]]['predecessors'].append(attack[0])
                    self.di_graph.add_edge(attack[0], attack[1])
        sorted_arguments = {key: self.arguments[key] for key in natsorted(self.arguments.keys())}
        self.arguments = sorted_arguments


    # build grounded extension based on Nofal et al.
    def get_grounded_ext(self):
        grounded_labelling = Labelling(self)
        undec_pre = {arg: len(value['predecessors']) for arg, value in self.arguments.items()}
        to_be_in = deque(arg for arg, count in undec_pre.items() if count == 0)

        while to_be_in:
            current_node = to_be_in.popleft()
            grounded_labelling.label_dict[current_node]['label'] = "in"

            successors = self.arguments[current_node]['successors']
            for attacker in successors:
                if grounded_labelling.label_dict[attacker]['label'] != 'out':
                    grounded_labelling.label_dict[attacker]['label'] = "out"
                    for potential_to_be_in in self.arguments[attacker]['successors']:
                        if grounded_labelling.label_dict[potential_to_be_in]['label'] == 'undec':
                            undec_pre[potential_to_be_in] -= 1
                            if undec_pre[potential_to_be_in] == 0:
                                to_be_in.append(potential_to_be_in)

        return [node for node, lab in grounded_labelling.label_dict.items() if lab['label'] == 'in']