#!/usr/bin/env python3
"""admissibility_solver.py – Single-core Backtracking Solver for Admissibility

Uses backtracking search with heuristics or specific task-based logic
to determine admissibility status based on the provided task parameter.

Features:
* Command-line interface for specifying AF file, query, task, time limit, etc.
* Task-based algorithm selection:
    - DS*/DC-ID: Grounded Extension check only.
    - DS-ST: Grounded check + SCC Reinstatement heuristic.
    - DC*: Grounded check + Backtracking search (with preprocessing).
* Always performs Grounded Extension check first.
* Optional Numba acceleration for performance-critical parts (conflict check, defect calculation, grounded).
* Preprocessing steps for DC tasks: Grounded Extension, SCC reduction, k-hop Cone Extraction.
* Optional progress snapshots (`--progress`).
* Graceful handling of missing NumPy/Numba with pure-Python fallbacks.

**Note:** Backtracking search is used for DC* tasks (excluding DC-ID).
Parameters related to MCTS (`--eps`, `--uct-c`) are not applicable.
"""
from __future__ import annotations

import argparse
import functools
import itertools
import logging
import re
import signal
import sys
import time
import random
from typing import Any, Dict, List, Set, Tuple, Optional # Added Optional

# --- Constants for Status ---
STATUS_IN = "YES"
STATUS_OUT = "NO"

# Using numpy only if available for performance.
# Script remains functional (but slower) without it.
try:
    import numpy as np
    HAVE_NUMPY = True
except ModuleNotFoundError:
    HAVE_NUMPY = False
    # Define dummy np object for type hinting consistency when numpy is missing
    class DummyNp:
        uint64 = int; int64 = int; ndarray = list; bool = bool
        def zeros(*args, **kwargs): return [0] * args[0]
        def array(*args, **kwargs): return list(args[0])
        def cumsum(a, out=None):
            total = 0
            res = []
            if out is not None: # Simulate out parameter if possible
                # Ensure 'out' is list-like and has correct size
                if isinstance(out, list) and len(out) == len(a) + 1:
                    out[0] = 0 # CSR indptr starts with 0
                    for i, x in enumerate(a):
                        total += x
                        out[i+1] = total
                    return out
                else: # Cannot simulate 'out' correctly
                    for x in a: total += x; res.append(total)
                    return res # Return a new list
            else:
                for x in a: total += x; res.append(total)
                return res
        def empty(*args, **kwargs): return [None] * args[0]
        def full(*args, **kwargs): return [kwargs.get('fill_value',0)] * args[0]
        def where(cond):
             # Special handling for grounded_csr usage: np.where(IN_bits_np == 1)[0]
            if isinstance(cond, list): # Simulate for list-based _grounded_csr fallback
                # Check if it's a list of 0s and 1s
                if all(x in (0, 1) for x in cond):
                     return ([i for i, x in enumerate(cond) if x == 1],) # Return tuple like numpy
            return ([],) # Default empty tuple like numpy's where

        def float64(*args, **kwargs): return float(*args)
        def all(a):
             # Check if all elements are zero (or equivalent False)
            if isinstance(a, list): return all(x == 0 for x in a)
            # Basic fallback for other iterables (might not match numpy exactly)
            try: return all(x == 0 for x in iter(a))
            except TypeError: return False # Not iterable or comparable
        uint8 = int # Add dummy type for bool array in grounded
        int64 = int
        bool = bool
        def fill(self, value): # Mock fill method for numpy arrays
            if isinstance(self, list):
                for i in range(len(self)): self[i] = value

    np = DummyNp()
    # Add fill method to list prototype - VERY HACKY, avoids changing seen.fill(False) later
    # This is generally bad practice, but done here to minimize code changes per request.
    # Consider replacing seen.fill(False) with `seen = [False] * n` if this causes issues.
    try:
        list.fill = DummyNp.fill
    except AttributeError: # Might fail in some environments
        logging.warning("Could not add mock 'fill' method to list type.")

    logging.error("NumPy not found. Performance will be significantly degraded.")

# --- Define Global Helper Functions based on NumPy/List ---
# These abstract away the data structure type (numpy array or list)

# Default values (Python list fallbacks)
bits_copy_global = list
# Adjusted get_bit_global to handle W correctly for lists
get_bit_global = lambda arr, idx, W: (arr[idx // 64] >> (idx % 64)) & 1 if (idx >= 0 and W > 0 and idx // 64 < len(arr) and idx // 64 < W) else 0
set_bit_inplace_np = None
restore_bit_inplace_np = None
set_bit_inplace_list = None
restore_bit_inplace_list = None
bits_to_tuple_global = tuple
bitwise_not_global = lambda vec, W: [( (1<<(64))-1 ) ^ x for x in vec] # List fallback (assumes uint64 words)
arr_and_global = lambda a, b: [x & y for x, y in zip(a, b)] # List fallback
vec_is_zero_global = lambda vec: all(x==0 for x in vec) # List fallback
arr_or_global = lambda a, b: [x | y for x, y in zip(a, b)] # List fallback

if HAVE_NUMPY:
    logging.debug("NumPy detected, defining NumPy-based global helpers.")
    bits_copy_global = lambda b: b.copy()
    # Adjusted get_bit_global for numpy to handle W correctly
    get_bit_global = lambda arr, idx, W: (arr[idx // 64] >> np.uint64(idx % 64)) & np.uint64(1) if (idx>=0 and W > 0 and idx//64 < arr.shape[0] and idx//64 < W) else np.uint64(0)
    def set_bit_inplace_np(arr: np.ndarray, idx: int, W: int):
        w, b = divmod(idx, 64)
        if w < W and w < arr.shape[0]: arr[w] |= (np.uint64(1) << np.uint64(b))
    def restore_bit_inplace_np(arr: np.ndarray, idx: int, W: int):
        w, b = divmod(idx, 64)
        if w < W and w < arr.shape[0]: arr[w] &= ~(np.uint64(1) << np.uint64(b))
    bits_to_tuple_global = lambda b: tuple(b)
    bitwise_not_global = lambda vec, W=0: ~vec # NumPy handles bitwise NOT
    arr_and_global = lambda a, b: a & b # NumPy vectorized AND
    vec_is_zero_global = lambda vec: np.all(vec == 0) # NumPy check
    arr_or_global = lambda a, b: a | b # NumPy vectorized OR

# Define list versions only if needed (used if HAVE_NUMPY is False)
if not HAVE_NUMPY:
    logging.debug("NumPy not detected, defining List-based global helpers.")
    def set_bit_inplace_list(arr: list, idx: int, W: int):
        w, b = divmod(idx, 64)
        if w < W and w < len(arr): arr[w] |= (1 << b)
    def restore_bit_inplace_list(arr: list, idx: int, W: int):
        w, b = divmod(idx, 64)
        if w < W and w < len(arr): arr[w] &= ~(1 << b)

# --- Numba Setup (depends on NumPy) ---
try:
    if not HAVE_NUMPY:
        raise ModuleNotFoundError("Numba requires NumPy, which was not found.")
    from numba import njit, uint64, int64 # type: ignore
    HAVE_NUMBA = True
    logging.info("Numba found, JIT acceleration enabled for performance-critical functions.")
except ModuleNotFoundError as e:
    HAVE_NUMBA = False
    # Use dummy/real np type based on HAVE_NUMPY check above
    uint64 = np.uint64 if HAVE_NUMPY else int
    int64 = np.int64 if HAVE_NUMPY else int
    logging.warning(f"Numba not found or NumPy missing ({e}). Using pure Python fallbacks.")
    # Dummy njit decorator
    def njit(*args, **kwargs):
        def deco(fn):
            @functools.wraps(fn)
            def wrapper(*a, **kw): return fn(*a, **kw)
            return wrapper
        # Handle decorator call with or without arguments
        if args and callable(args[0]): return deco(args[0])
        else: return deco

# --- End Dependency Setup ---

# ─────────────────────────────────────────────────────────────────────────────
# Helper Function for Conflict Check (Numba accelerated if possible)
# ─────────────────────────────────────────────────────────────────────────────

# Numba helper for checking conflicts when adding 'a' to 'bits'
if HAVE_NUMBA and HAVE_NUMPY:
    @njit("boolean(uint64, uint64[:], uint64[:], uint64[:])", cache=True)
    def check_add_conflict_numba(a_unused: int, # 'a' index not needed if rows passed
                                 bits: np.ndarray,
                                 out_a_row: np.ndarray, # Row from out_bits for candidate 'a'
                                 in_a_row: np.ndarray   # Row from in_bits for candidate 'a'
                                 ) -> bool:
        """Checks if adding 'a' creates conflict with existing 'bits'."""
        W = bits.shape[0] # Get width from bits array
        # Ensure rows have same width as bits
        if out_a_row.shape[0] != W or in_a_row.shape[0] != W:
            # Cannot log from numba, returning True indicates error/conflict
            return True
        for k in range(W):
            # Check if a attacks bits (out_a_row & bits) OR bits attacks a (in_a_row & bits)
            if (out_a_row[k] & bits[k]) != np.uint64(0) or \
               (in_a_row[k] & bits[k]) != np.uint64(0):
                return True # Conflict found
        return False # No conflict
else:
    # Provide a Python fallback if Numba/NumPy not available
    def check_add_conflict_numba(a_unused: int, bits: list, out_a_row: list, in_a_row: list) -> bool:
        """Python fallback for conflict check."""
        W = len(bits)
        # Check for empty lists which might indicate an error earlier
        if W == 0: return False # No conflicts possible in empty set/graph context
        # Basic dimension check
        if len(out_a_row) != W or len(in_a_row) != W:
             # Logging happens outside Numba context
             # print(f"Error: Dimension mismatch in conflict check. bits W={W}, out_a W={len(out_a_row)}, in_a W={len(in_a_row)}")
             return True # Assume conflict on error

        for k in range(W):
            if (out_a_row[k] & bits[k]) != 0 or (in_a_row[k] & bits[k]) != 0:
                return True
        return False

# --- End Conflict Check Helper Definition ---

# ───────────────────────── popcount for uint64 ──────────────────────────
if HAVE_NUMBA:
    @njit("int64(uint64)", cache=True)
    def popcount_u64(x):
        """Counts set bits in a uint64 integer (Numba version)."""
        one = np.uint64(1) # Typed constant
        cnt = 0
        while x != 0:
            x = x & (x - one)
            cnt += 1
        return cnt
else:
    # Fallback using Python's built-in bit_count (available Python 3.10+)
    # or a simple loop for older versions.
    if hasattr(int, 'bit_count'):
        def popcount_u64(x):
            """Counts set bits in a uint64 integer (Python >= 3.10 fallback)."""
            try:
                # Ensure x is int; numpy uint64 might need conversion
                return int(x).bit_count()
            except (TypeError, ValueError):
                # logging.error(f"Could not convert value {x} of type {type(x)} to int for bit_count.")
                return 0 # Or raise an error? Returning 0 might hide issues.
    else: # Fallback for Python < 3.10
        def popcount_u64(x):
            """Counts set bits in a uint64 integer (Python < 3.10 fallback)."""
            try:
                x_int = int(x)
                cnt = 0
                while x_int > 0:
                    x_int &= (x_int - 1) # Clear the least significant bit set
                    cnt += 1
                return cnt
            except (TypeError, ValueError):
                 # logging.error(f"Could not convert value {x} of type {type(x)} to int for popcount.")
                 return 0

# ---------------------------------------------------------------------------
# Fast grounded extension (Numba CSR + queue)
# ---------------------------------------------------------------------------
# Define Numba type signatures based on availability
if HAVE_NUMBA:
    _grounded_csr_sig = "uint8[:](int64[:], int64[:], int64[:], int64[:])"
    @njit(_grounded_csr_sig, cache=True)
    def _grounded_csr(indptr: np.ndarray, indices: np.ndarray,
                      rev_indptr: np.ndarray, rev_indices: np.ndarray) -> np.ndarray:
        """Numba-accelerated grounded extension calculation using CSR format."""
        n = indptr.shape[0] - 1
        if n <= 0: return np.zeros(0, dtype=np.uint8)

        IN = np.zeros(n, dtype=np.uint8)  # 1=in, 0=undec/out
        OUT = np.zeros(n, dtype=np.uint8) # 1=out
        # Initialize atk_left: number of attackers NOT marked OUT yet
        atk_left = np.zeros(n, dtype=np.int64) # Initialize explicitly
        for i in range(n):
             # Need to check bounds for rev_indptr
             if i + 1 < rev_indptr.shape[0]:
                 atk_left[i] = rev_indptr[i+1] - rev_indptr[i]
             # Else atk_left remains 0, which is correct for nodes with no attackers

        # Initial queue of arguments with 0 attackers
        # Use a list as dynamic queue buffer inside Numba is efficient
        q = [i for i in range(n) if atk_left[i] == 0]
        q_head = 0
        while q_head < len(q):
            a = q[q_head]
            q_head += 1

            # Skip if already processed (marked IN or OUT)
            if IN[a] != 0 or OUT[a] != 0:
                continue

            # Mark 'a' as IN (since all its attackers are OUT)
            IN[a] = 1
            # Mark everything 'a' attacks as OUT
            # Check bounds for indptr access
            if a + 1 < indptr.shape[0]:
                for j in range(indptr[a], indptr[a + 1]):
                    # Check bounds for indices access
                    if j < indices.shape[0]:
                        v = indices[j]
                        # Check bounds for target node v
                        if 0 <= v < n:
                            if OUT[v] == 0: # Mark only if not already OUT
                                OUT[v] = 1
                                # Decrement attacker count for nodes attacked by 'v'
                                # Check bounds for indptr access for v
                                if v + 1 < indptr.shape[0]:
                                    for k in range(indptr[v], indptr[v + 1]):
                                        # Check bounds for indices access for v's targets
                                        if k < indices.shape[0]:
                                            tgt = indices[k]
                                            if 0 <= tgt < n: # Check bounds for target tgt
                                                 # Check bounds for atk_left access
                                                 if 0 <= tgt < n:
                                                    atk_left[tgt] -= 1
                                                    # If tgt becomes potentially IN, add to queue
                                                    if atk_left[tgt] == 0 and IN[tgt] == 0 and OUT[tgt] == 0:
                                                        q.append(tgt)
        # Return array where 1 indicates IN
        return IN
else:
    # Define a dummy placeholder if Numba isn't available
    def _grounded_csr(*args, **kwargs):
        # Should not be called if HAVE_NUMBA is False
        logging.error("Internal Error: _grounded_csr called without Numba.")
        return None # Or raise an error

# ---------------------------------------------------------------------------
# Compact CSR helpers (only if NumPy is available)
# ---------------------------------------------------------------------------
if HAVE_NUMPY:
    def to_csr(adj_lists: List[List[int]]) -> Tuple[np.ndarray, np.ndarray]:
        """Convert list-of-lists adjacency to CSR (indptr, indices)."""
        n = len(adj_lists)
        if n == 0:
            return np.array([0], dtype=np.int64), np.array([], dtype=np.int64)

        # Ensure lists are valid before processing
        counts = np.array([len(r) for r in adj_lists], dtype=np.int64)
        indptr = np.empty(n + 1, dtype=np.int64)
        indptr[0] = 0
        np.cumsum(counts, out=indptr[1:]) # Use numpy's cumsum
        total_edges = indptr[-1]
        # Handle case where total_edges might be non-integer if cumsum failed (unlikely)
        if not isinstance(total_edges, (int, np.integer)): total_edges = 0
        indices = np.empty(int(total_edges), dtype=np.int64)

        # Efficiently fill indices
        if total_edges > 0:
            try:
                # Ensure all elements in adj_lists are iterable and contain integers
                flat_list = []
                valid = True
                for row in adj_lists:
                    if not isinstance(row, list): valid=False; break
                    if not all(isinstance(x, int) for x in row): valid=False; break
                    flat_list.extend(row)

                if valid and len(flat_list) == total_edges:
                    indices[:] = flat_list
                else:
                    # Fallback if lengths mismatch or data invalid
                    logging.warning("CSR creation issue (length/type), using slower fill.")
                    k = 0
                    for row in adj_lists:
                         if isinstance(row, list) and all(isinstance(x, int) for x in row):
                              row_len = len(row)
                              if k + row_len <= total_edges:
                                   indices[k : k + row_len] = row
                                   k += row_len
                              else: break # Avoid writing past allocated space
            except Exception as e:
                logging.error(f"Error during CSR index filling: {e}. Using safe loop.")
                k = 0
                for row in adj_lists:
                    try:
                        row_len = len(row)
                        if k + row_len <= total_edges:
                            indices[k : k + row_len] = row
                            k += row_len
                        else: break # Avoid writing past allocated space
                    except (TypeError, ValueError):
                        logging.error(f"Skipping invalid row data during CSR fill: {row}")
        return indptr, indices
else:
    # Define a dummy to_csr if NumPy is not available, so ensure_csr doesn't fail
    def to_csr(adj_lists: List[List[int]]) -> Tuple[Any, Any]:
        logging.debug("NumPy not available, cannot create CSR format.")
        return None, None

###############################################################################
# Hard timeout decorator (POSIX only)
###############################################################################

def hard_timeout(sec: int = 580):
    """Raise ``TimeoutError`` once *sec* seconds have elapsed (wall clock)."""
    def deco(fn):
        # Check if running on a non-POSIX system (like Windows)
        if not hasattr(signal, "SIGALRM"):
            logging.warning("signal.SIGALRM not available on this system. Hard timeout disabled.")
            return fn # Return the original function without timeout

        @functools.wraps(fn)
        def wrap(*a, **kw):
            def _handle_timeout(signum, frame):
                logging.error(f"Hard timeout of {sec} seconds triggered.")
                raise TimeoutError(f"Operation hard-timed out after {sec} seconds")

            old_handler = signal.signal(signal.SIGALRM, _handle_timeout)
            # Ensure sec is non-negative before calling alarm
            signal.alarm(max(0, int(sec))) # Schedule the alarm

            try:
                result = fn(*a, **kw)
            finally:
                # Cancel any pending alarm and restore the previous handler
                signal.alarm(0)
                signal.signal(signal.SIGALRM, old_handler)
            return result
        return wrap
    return deco

###############################################################################
# Logging helper
###############################################################################

def setup_logging(debug: bool, lvl: str):
    log_level = logging.DEBUG if debug else getattr(logging, lvl.upper(), logging.INFO)
    # Use basicConfig only if no handlers are configured yet
    # Use force=True in Python 3.8+ to override existing config if needed
    kwargs = {'level': log_level,
              'format': "[%(asctime)s %(levelname)s] %(message)s",
              'datefmt': "%H:%M:%S",
              'stream': sys.stderr}
    if sys.version_info >= (3, 8): kwargs['force'] = True

    if not logging.root.handlers or sys.version_info >= (3, 8):
         logging.basicConfig(**kwargs)
    else:
         # If handlers exist and force=True is not available, just set the level
         logging.getLogger().setLevel(log_level)
         # Optionally, reconfigure handlers if they exist but format/level is wrong
         # (More complex, usually basicConfig with force=True is preferred)

    logging.info(f"Logging initialized at level: {logging.getLevelName(log_level)}")


###############################################################################
# AF container & parser
###############################################################################

a_pair = re.compile(r"^(\d+)\s+(\d+)$")

class AF:
    """Represents an Argumentation Framework."""
    __slots__ = ("n", "out_edges", "in_edges", "orig2int", "int2orig",
                 "_out_indptr", "_out_indices", "_in_indptr", "_in_indices",
                 "_csr_ready")
    n: int
    out_edges: List[List[int]] # List of outgoing neighbors for each internal node ID
    in_edges: List[List[int]]  # List of incoming neighbors for each internal node ID
    orig2int: Optional[Dict[int, int]] # Map from original ID (file) to internal ID (0..n-1)
    int2orig: Optional[List[int]]      # Map from internal ID to original ID (file)
    _out_indptr: Optional[Any] # CSR format (NumPy array or None)
    _out_indices: Optional[Any]
    _in_indptr: Optional[Any]
    _in_indices: Optional[Any]
    _csr_ready: bool

    def __init__(self, n: int):
        if n < 0: raise ValueError("Number of arguments 'n' cannot be negative")
        self.n = n
        self.out_edges = [[] for _ in range(n)]
        self.in_edges = [[] for _ in range(n)]
        self.orig2int = None
        self.int2orig = None
        self._out_indptr = None
        self._out_indices = None
        self._in_indptr = None
        self._in_indices = None
        self._csr_ready = False

    def add_edge(self, u: int, v: int):
        """Adds a directed edge from u to v (using internal indices)."""
        if 0 <= u < self.n and 0 <= v < self.n:
            # Avoid adding duplicate edges if performance is critical,
            # but standard parsing usually doesn't require this check.
            self.out_edges[u].append(v)
            self.in_edges[v].append(u)
            self._csr_ready = False # Invalidate CSR if edges change
        else:
            logging.warning(f"Attempted to add invalid edge ({u}, {v}) for n={self.n}. Ignored.")

    def ensure_csr(self):
        """Builds CSR arrays if NumPy is available and they haven't been built yet."""
        if not self._csr_ready and HAVE_NUMPY:
            logging.debug("Building CSR representation for AF.")
            try:
                 self._out_indptr, self._out_indices = to_csr(self.out_edges)
                 self._in_indptr,  self._in_indices  = to_csr(self.in_edges)
                 # Check if to_csr returned valid arrays (it returns None if NumPy isn't present)
                 if self._out_indptr is not None and self._in_indptr is not None:
                     self._csr_ready = True
                     logging.debug("CSR representation built successfully.")
                 else:
                     # This case should ideally not be reached if HAVE_NUMPY is true,
                     # but handles potential edge cases in to_csr returning None unexpectedly.
                     logging.warning("CSR creation failed unexpectedly even with NumPy available.")
                     self._csr_ready = False
            except Exception as e:
                 logging.error(f"Failed to build CSR representation: {e}")
                 self._csr_ready = False
                 self._out_indptr, self._out_indices = None, None
                 self._in_indptr,  self._in_indices  = None, None
        elif self._csr_ready:
              logging.debug("CSR representation already available.")
        elif not HAVE_NUMPY:
              logging.debug("NumPy not available, cannot build CSR representation.")


def parse_af(path: str) -> Optional[AF]:
    """Read an AF in the *.apx*‑style format used in ICCMA."""
    logging.info(f"Parsing AF from: {path}")
    n_decl = -1 # Declared number of arguments
    m_decl = None # Declared number of attacks
    attacks: List[Tuple[int, int]] = []
    ids: Set[int] = set() # Set of original argument IDs found in attacks

    try:
        with open(path, encoding="utf-8") as f:
            line_num = 0
            header_found = False
            # Find header line (p af n [m])
            while not header_found:
                line_num += 1
                raw = f.readline()
                if not raw: break # End of file
                line = raw.strip()
                if not line or line.startswith(("c", "#")): continue # Skip comments/empty

                header = line.split()
                if len(header) >= 3 and header[0] == "p" and header[1] == "af":
                    try:
                        n_decl = int(header[2])
                        if n_decl < 0: raise ValueError("Header 'n' cannot be negative.")
                        if len(header) >= 4:
                            m_decl = int(header[3])
                            if m_decl < 0: raise ValueError("Header 'm' cannot be negative.")
                        header_found = True
                        logging.debug(f"Parsed header: n={n_decl}, m={m_decl if m_decl is not None else 'N/A'}")
                    except ValueError as e:
                        raise ValueError(f"Invalid number(s) in header line {line_num}: '{line}'. Error: {e}")
                else:
                    raise ValueError(f"Expected 'p af n [m]' header, found '{line}' at line {line_num}.")

            if not header_found:
                raise ValueError("AF file lacks a valid 'p af n [m]' header line.")

            # Read attack lines
            for raw in f:
                line_num += 1
                line = raw.strip()
                if not line or line.startswith(("c", "#")): continue

                m = a_pair.match(line) # Match "int int" format
                if not m:
                    logging.warning(f"Skipping malformed edge line {line_num}: '{line}'")
                    continue
                try:
                    u, v = map(int, m.groups())
                    if u < 0 or v < 0: # Basic validation
                        logging.warning(f"Skipping edge line {line_num} with negative argument ID(s): '{line}'")
                        continue
                    ids.update((u, v))
                    attacks.append((u, v))
                except ValueError:
                    logging.warning(f"Skipping edge line {line_num} with non-integer argument ID(s): '{line}'")
                    continue

    except FileNotFoundError:
        logging.error(f"File not found: {path}")
        return None
    except ValueError as e:
        logging.error(f"Error parsing AF file {path}: {e}")
        return None
    except Exception as e:
        logging.error(f"An unexpected error occurred during parsing {path}: {e}")
        return None

    # --- Post-parsing processing ---
    if n_decl == -1: # Should be caught earlier
        logging.error("Parsing finished without finding a header.")
        return None

    # Handle empty AF cases
    if n_decl == 0 and not ids and not attacks:
        logging.info("Parsed an empty AF (n=0, no attacks).")
        af = AF(0)
        af.orig2int = {}
        af.int2orig = []
        return af

    # Determine actual arguments from attacks
    id_list_sorted = sorted(list(ids))
    n_actual_in_attacks = len(id_list_sorted)

    # Decide the actual number of nodes 'n' for the AF object.
    # Use n_decl if provided and greater than max found ID, otherwise use max found ID + 1 or n_actual?
    # Simpler: Base 'n' on the highest ID actually *seen* in attacks or declared, whichever is larger.
    # This handles isolated nodes declared in 'n' but not present in attacks.
    # However, ICCMA format doesn't guarantee IDs are 0..n-1 or 1..n.
    # Sticking to the previous logic: AF size based on unique IDs *in attacks*.
    # This means isolated nodes declared in header but not in attacks are ignored.

    n_for_af = n_actual_in_attacks # How many nodes will have adjacency lists

    if n_for_af == 0 and n_decl > 0:
         # Header declared nodes, but none are involved in attacks.
         # Create an AF with n_decl isolated nodes. Need mapping.
         logging.info(f"Header declared n={n_decl} but no attacks found. Creating AF with {n_decl} conceptual nodes, but 0 interacting nodes.")
         # The current AF class represents interacting nodes. An empty AF is appropriate.
         af = AF(0)
         af.orig2int = {}
         af.int2orig = []
         # Note: Querying an isolated node would fail later as it's not in orig2int.
         return af
    elif n_for_af == 0 and n_decl == 0:
         # Already handled by empty AF check
         pass # Fall through to create AF(0)

    # Check consistency between declared n and found n
    if n_decl > 0 and n_decl != n_actual_in_attacks:
        # Find max ID actually used
        max_id_in_attacks = max(id_list_sorted) if id_list_sorted else -1
        # A common format is args are 1..n_decl. Check if this seems likely.
        potential_range_n = max(n_decl, max_id_in_attacks + 1 if max_id_in_attacks >=0 else n_decl)
        if potential_range_n > n_actual_in_attacks:
             logging.warning(
                 f"Header declared n={n_decl}, but only {n_actual_in_attacks} unique argument IDs found in attacks. "
                 f"Implies {n_decl - n_actual_in_attacks} isolated nodes or non-contiguous IDs. "
                 f"Solver internal representation uses n={n_for_af} based on interacting arguments."
             )
        # If n_decl < n_actual_in_attacks, the header is simply wrong/misleading.
        elif n_decl < n_actual_in_attacks:
             logging.warning(
                 f"Header declared n={n_decl}, but {n_actual_in_attacks} unique argument IDs found in attacks. Header count is too low. "
                 f"Using n={n_for_af} based on interacting arguments."
              )

    # Create mapping and AF object using nodes found in attacks
    o2i = {orig_id: internal_idx for internal_idx, orig_id in enumerate(id_list_sorted)}
    af = AF(n_for_af)
    af.orig2int = o2i
    af.int2orig = id_list_sorted # Store the original IDs corresponding to internal indices 0..n_for_af-1

    # Add edges using internal indices
    num_added_edges = 0
    for u_orig, v_orig in attacks:
        # Map original IDs to internal 0-based indices
        # Check if they exist in the map (they should if id_list_sorted was built from attacks)
        if u_orig in o2i and v_orig in o2i:
            u_internal, v_internal = o2i[u_orig], o2i[v_orig]
            af.add_edge(u_internal, v_internal)
            num_added_edges += 1
        else:
             # This case should not happen if logic is correct
             logging.error(f"Internal error: Attack edge ({u_orig}, {v_orig}) contains ID not found in mapping.")

    # Check attack count consistency
    if m_decl is not None and m_decl != num_added_edges:
        logging.warning(f"Header declared m={m_decl} attacks, but {num_added_edges} valid attacks were processed.")

    logging.info(f"Successfully parsed AF: {af.n} nodes (in attacks), {num_added_edges} attacks.")
    return af


###############################################################################
# Simple Kosaraju SCC decomposition (iterative)
###############################################################################
def strongly_connected_components(af: AF) -> List[List[int]]:
    """Finds Strongly Connected Components using Kosaraju's algorithm."""
    n = af.n
    if n == 0: return []
    order: List[int] = [] # Stores finish order in first DFS pass
    # Use NumPy bool array if available, else list
    # Ensure seen is created with correct size n
    seen: Any = np.zeros(n, dtype=np.bool_) if HAVE_NUMPY else [False] * n

    # First pass: DFS on original graph to get post-order
    for v_start in range(n):
        if not seen[v_start]:
            stack = [(v_start, iter(af.out_edges[v_start]))] # (node, child_iterator)
            # path_nodes = {v_start} # Nodes currently in DFS path (not strictly needed for iterative)

            while stack:
                u, children_iter = stack[-1]
                # Check bounds for seen access
                if not (0 <= u < n):
                    stack.pop() # Invalid node index on stack, skip
                    continue
                seen[u] = True # Mark node as visited

                try:
                    v = next(children_iter) # Get next child
                    # Check bounds for v and seen access
                    if 0 <= v < n and not seen[v]: # If child not visited, push to stack
                        stack.append((v, iter(af.out_edges[v])))

                except StopIteration: # Finished exploring children of u
                    stack.pop()
                    # path_nodes.remove(u) # Not needed
                    order.append(u) # Add to post-order list

    # Second pass: DFS on transpose graph in reverse post-order
    if HAVE_NUMPY and isinstance(seen, np.ndarray):
        seen.fill(False) # Reset seen array efficiently if numpy
    else:
        seen = [False] * n # Reset list-based seen array

    comp_list: List[List[int]] = []
    for v_start in reversed(order):
         # Check bounds for v_start and seen access
        if 0 <= v_start < n and not seen[v_start]:
            comp: List[int] = []
            q = [v_start] # Stack for DFS on transpose graph
            seen[v_start] = True
            while q:
                u = q.pop()
                 # Check bounds for u access
                if not (0 <= u < n): continue # Invalid index popped, skip

                comp.append(u)
                # Explore neighbors in the *transpose* graph (using in_edges)
                for v in af.in_edges[u]:
                    # Check bounds for v and seen access
                    if 0 <= v < n and not seen[v]:
                        seen[v] = True
                        q.append(v)
            if comp:
                comp_list.append(sorted(comp)) # Sort for consistent output

    logging.debug(f"Found {len(comp_list)} SCCs.")
    return comp_list


###############################################################################
# Grounded extension (exact)
###############################################################################

def grounded_extension(af: AF) -> Set[int]:
    """
    Calculates the least fixed-point grounded extension.
    Uses NumPy/Numba acceleration if available, otherwise pure Python.
    Returns a set of internal argument indices.
    """
    n = af.n
    if n == 0: return set()

    # Ensure CSR is built before trying Numba path (if NumPy available)
    if HAVE_NUMPY:
        af.ensure_csr()

    # Try Numba path first if available and CSR is ready
    if HAVE_NUMBA and af._csr_ready and af._out_indptr is not None and \
       af._out_indices is not None and af._in_indptr is not None and \
       af._in_indices is not None:
        try:
            logging.debug("Using Numba-accelerated grounded extension.")
            # Call the Numba JIT function (_grounded_csr returns np.uint8 array)
            IN_bits_np = _grounded_csr(
                af._out_indptr, af._out_indices,
                af._in_indptr, af._in_indices,
            )
            # Check return type from Numba function
            if IN_bits_np is None or not isinstance(IN_bits_np, np.ndarray):
                 raise TypeError("Numba _grounded_csr did not return a numpy array.")

            # Convert numpy uint8 array (0/1) to set of indices where value is 1
            # Use np.where for efficiency
            grounded_indices = np.where(IN_bits_np == 1)[0]
            # Convert numpy array of indices to a set of Python ints
            return set(grounded_indices.astype(int))

        except Exception as e:
            logging.warning(f"Numba grounded extension failed: {e}. Falling back to pure Python.")
            # Fall through to Python implementation

    # Pure Python fallback implementation (LFP calculation)
    logging.debug("Using pure Python grounded extension.")
    IN: Set[int] = set()
    OUT: Set[int] = set()
    # Pre-calculate attackers for efficiency (list of sets/lists)
    attackers_list = [set(af.in_edges[i]) for i in range(n)]
    # Pre-calculate attacked nodes (list of sets/lists)
    attacked_list = [set(af.out_edges[i]) for i in range(n)]

    changed = True
    while changed:
        changed = False
        newly_IN: Set[int] = set()
        newly_OUT: Set[int] = set()

        # Check for newly IN arguments (all attackers are OUT)
        for i in range(n):
            if i not in IN and i not in OUT:
                # Check if all attackers are OUT
                # Use pre-calculated attackers_list[i]
                if all(attacker in OUT for attacker in attackers_list[i]):
                    newly_IN.add(i)
                    changed = True # Found a newly IN argument

        # Add newly IN arguments
        IN.update(newly_IN)

        # Check for newly OUT arguments (attacked by newly IN)
        for i in newly_IN: # Only need to check args attacked by newly_IN ones
            # Use pre-calculated attacked_list[i]
            for attacked in attacked_list[i]:
                if attacked not in OUT:
                    newly_OUT.add(attacked)
                    changed = True # Marking something OUT is a change

        OUT.update(newly_OUT)

        # Consistency check: Remove from IN any argument that became OUT
        # (Should not happen if logic is correct, but ensures OUT overrides IN)
        in_became_out = IN.intersection(OUT)
        if in_became_out:
            IN.difference_update(in_became_out)
            logging.debug(f"Removed {len(in_became_out)} arguments from IN because they became OUT.")
            # No need to set changed=True here, loop continues if newly_IN/newly_OUT occurred

    logging.debug(f"Grounded set size: {len(IN)}")
    return IN

###############################################################################
# SCC Reinstatement Heuristic (Adapted)
###############################################################################
def run_scc_reinstatement(af: AF, q_internal: int) -> str:
    """
    Checks if external attackers of target's SCC are defeated by Grounded.
    Adapted to use the AF object and internal index q_internal.
    """
    # Basic checks
    if not (0 <= q_internal < af.n):
        logging.warning(f"SCC Reinstatement: Query {q_internal} out of bounds for AF size {af.n}")
        return STATUS_OUT

    # 1. Calculate Grounded Extension
    try:
        grounded_set = grounded_extension(af)
    except Exception as e:
        logging.error(f"SCC Reinstatement: Failed to compute grounded extension: {e}")
        return STATUS_OUT # Cannot proceed without grounded

    # 2. Check if query is already in Grounded
    if q_internal in grounded_set:
        logging.debug(f"SCC Reinstatement: Query {q_internal} is IN grounded set.")
        return STATUS_IN

    # 3. Find SCC of the target argument
    try:
        sccs = strongly_connected_components(af)
        if not sccs: # Handle case of no SCCs found (e.g., error or empty graph processed)
            logging.warning("SCC Reinstatement: No SCCs found. Cannot determine target SCC.")
            return STATUS_OUT

        # Build node -> SCC mapping (using the SCC component list itself as the 'set')
        node_to_scc_list_ref: Dict[int, List[int]] = {}
        for comp_list in sccs:
            for node in comp_list:
                 if 0 <= node < af.n: # Bounds check
                     node_to_scc_list_ref[node] = comp_list

    except Exception as e:
        logging.error(f"SCC Reinstatement: Failed during SCC decomposition: {e}")
        return STATUS_OUT

    # Check if query was found in any SCC
    if q_internal not in node_to_scc_list_ref:
        # This might happen if q_internal was isolated and SCC only returned non-trivial ones,
        # or if there was an error mapping.
        # If isolated, it has no attackers, trivially defended? Heuristic definition needed.
        # Assuming if not in an SCC (and not grounded), it's OUT.
        logging.warning(f"SCC Reinstatement: Query {q_internal} not found in any SCC component.")
        # If it has no attackers at all, is it IN? Grounded would have caught this.
        # If it has attackers, but they are all outside its (trivial) SCC, proceed.
        scc_of_target_set = {q_internal} # Treat as trivial SCC if not found otherwise
        # Let's refine: if q_internal has no attackers it should be in grounded.
        # If it has attackers, and wasn't found in an SCC list, something is wrong.
        # Safest is OUT if not mapped.
        # Re-evaluate: if SCC func returns only non-trivial, isolated nodes need handling.
        # Let's assume SCC func returns all components, even singletons. If still not found -> error.
        # If the SCC function *might* omit singletons: Check if q_internal has attackers.
        # If af.in_edges[q_internal] is empty, it should be in grounded. If it's not -> contradiction?
        # If af.in_edges[q_internal] is not empty, and it's not in SCC list -> map error or SCC bug.
        # --> Sticking with OUT if not found in the map from the SCC result.
        return STATUS_OUT

    # Get the set representation of the target's SCC
    scc_of_target_set = set(node_to_scc_list_ref[q_internal])

    # 4. Find External Attackers of the SCC
    external_attackers = set()
    for node_in_scc in scc_of_target_set:
         # Check bounds just in case
        if 0 <= node_in_scc < af.n:
             for attacker in af.in_edges[node_in_scc]:
                 # Check if attacker is outside the target's SCC
                 if attacker not in scc_of_target_set:
                      # Check bounds of attacker
                     if 0 <= attacker < af.n:
                         external_attackers.add(attacker)
                     else: logging.warning(f"SCC Reinstatement: Attacker {attacker} out of bounds.")


    # 5. Check if all external attackers are defeated by the grounded set
    # An attacker is defeated if it is *in* the grounded set (contrary to original heuristic comment?)
    # Standard admissibility requires the *set* (containing the query) to defend *itself*.
    # This heuristic seems different: check if the *environment* (Grounded) defeats external threats.
    # Let's assume the heuristic means: Are all external attackers *NOT* in the Grounded set?
    # (i.e., they are either OUT or UNDEC relative to Grounded)
    # --> Re-reading the provided heuristic: "defeated by Grounded" likely means the attacker is *attacked* by Grounded.
    # --> Simpler interpretation: Check if any external attacker is *itself* in the Grounded set.
    # If an external attacker *is* in G, it's a valid threat, so the target is OUT.
    # If *no* external attacker is in G, the target *might* be IN (heuristic assumes SCC is internally consistent).

    # Revision based on common sense: External attackers must be defeated.
    # Defeated means attacked by the grounded set OR being outside the grounded set? Let's use the simpler "not in G" check.
    # If any external attacker IS IN the grounded set, the query is definitely OUT.

    any_attacker_in_grounded = False
    for ext_attacker in external_attackers:
        if ext_attacker in grounded_set:
            any_attacker_in_grounded = True
            logging.debug(f"SCC Reinstatement: External attacker {ext_attacker} of SCC is IN grounded.")
            break

    # Result: IN if no external attacker is in Grounded, OUT otherwise.
    result = STATUS_OUT if any_attacker_in_grounded else STATUS_IN
    logging.debug(f"SCC Reinstatement: Query {q_internal}. External attackers: {external_attackers}. Grounded: {grounded_set}. Result: {result}")
    return result


###############################################################################
# k‑hop cone (vertex level)
###############################################################################

def k_hop_cone(af: AF, q: int, k: int) -> Set[int]:
    """Calculates the k-hop neighborhood (incoming and outgoing paths) around q."""
    if not (0 <= q < af.n):
        logging.error(f"Query node {q} is out of bounds for AF size {af.n}")
        return set()
    if k < 0:
        logging.warning("k-hop cone requested with negative k, returning just the query node.")
        return {q}

    visited: Set[int] = {q}
    frontier: Set[int] = {q} # Nodes at the current distance level

    for i in range(k):
        if not frontier: # Stop if frontier becomes empty
            logging.debug(f"k-hop cone expansion stopped early at level {i+1} as frontier became empty.")
            break
        next_frontier = set()
        for node in frontier:
             # Check bounds for node access
            if not (0 <= node < af.n): continue # Skip invalid node in frontier

            # Combine neighbors from both incoming and outgoing edges
            neighbors = itertools.chain(af.out_edges[node], af.in_edges[node])
            for neighbor in neighbors:
                 # Basic bounds check on neighbor index
                if 0 <= neighbor < af.n:
                    if neighbor not in visited:
                        visited.add(neighbor)
                        next_frontier.add(neighbor)
                else:
                     # This indicates an issue with the AF edge data if neighbor index invalid
                     logging.warning(f"Neighbor {neighbor} of node {node} is out of bounds ({af.n}). Skipping.")
        frontier = next_frontier # Move to the next level

    logging.debug(f"{k}-hop cone size around query {q}: {len(visited)}")
    return visited


# ---------------------------------------------------------------------------
# Word‑vector defect score (Numba accelerated)
# ---------------------------------------------------------------------------
# Calculates admissibility defect: (internal conflicts + undefended attacks)
# Uses Numba JIT version if available, otherwise pure Python fallback.

if HAVE_NUMBA:
    # Numba signature definition
    _ad_defect_vec_sig = "int64(uint64[:], uint64[:,:], uint64[:,:])"
    @njit(_ad_defect_vec_sig, cache=True, fastmath=True) # Enable fastmath for potential speedup
    def ad_defect_vec(bits: np.ndarray, out_bits: np.ndarray, in_bits: np.ndarray) -> int64:
        """Calculates admissibility defect (Numba version)."""
        # Check if dimensions match conceptually
        m, W = out_bits.shape # m = number of nodes in cone, W = word vector width
        if bits.shape[0] != W or in_bits.shape != (m, W):
             # Cannot log, return large number indicating error?
             # Or rely on caller ensuring correct shapes. Let's assume correct shapes.
             # If W is 0, return 0 defect?
             if W == 0 : return int64(0)
             # If shapes mismatch significantly, maybe return -1?
             # For now, proceed assuming shapes are compatible if W>0.

        if m == 0 or W == 0 : return int64(0) # Defect is 0 for empty set/graph

        # Calculate defenders: nodes attacked by the set 'bits'
        defenders = np.zeros(W, dtype=np.uint64)
        # Iterate through potential nodes, check bit, then proceed.
        for i in range(m):
             w, b = divmod(i, 64)
             # Check if bit 'i' is set in 'bits' (ensure w is within bounds)
             if w < W and (bits[w] >> np.uint64(b)) & np.uint64(1):
                 # If arg i is IN, add its attacked nodes (out_bits[i]) to defenders
                 # Array OR operation is efficient in Numba
                 defenders |= out_bits[i] # Or across the whole row vector

        # Calculate conflicts and undefended attacks
        conflicts = np.int64(0)
        undef = np.int64(0)

        # Precompute bitwise NOT of defenders
        not_defenders = ~defenders

        for i in range(m):
             w, b = divmod(i, 64)
             # Check if bit 'i' is set in 'bits'
             if w < W and (bits[w] >> np.uint64(b)) & np.uint64(1):
                 # If arg i is IN:
                 # Use precomputed rows for efficiency
                 out_i = out_bits[i]
                 in_i = in_bits[i]
                 for k in range(W):
                     # 1. Conflicts: nodes attacked BY i (out_i) that are also IN (bits)
                     conflicts += popcount_u64(out_i[k] & bits[k])
                     # 2. Undefended: nodes that attack i (in_i) that are NOT attacked BY the set (not_defenders)
                     undef += popcount_u64(in_i[k] & not_defenders[k])

        # Defect = Conflicts + Undefended Attacks
        return conflicts + undef
else:
    # Pure Python fallback for ad_defect_vec
    def ad_defect_vec(bits: list, out_bits: list[list], in_bits: list[list]) -> int:
        """Calculates admissibility defect (Pure Python version)."""
        # Check if out_bits or in_bits are empty or malformed
        if not out_bits or not isinstance(out_bits[0], list): m = 0
        else: m = len(out_bits)

        if m == 0: return 0
        # Determine W from bits, handle if bits is empty
        if not bits: W = 0 # Cannot determine W if bits is empty
        else: W = len(bits)

        # If W is 0 but m>0, we might need to infer W or error
        if W == 0 and m > 0: W = (m + 63) // 64 # Infer W based on m
        if W == 0: return 0 # Still 0 W, defect is 0

        # Basic dimension checks for list mode
        if len(bits) != W or len(in_bits) != m or not all(len(row) == W for row in out_bits if row is not None) or not all(len(row) == W for row in in_bits if row is not None):
            logging.error("Dimension mismatch in ad_defect_vec (Python list mode)")
            return 999999 # Return large number to indicate error

        defenders = [0] * W
        conflicts = 0
        undef = 0

        set_indices = [] # Find indices of nodes IN the set 'bits'
        for i in range(m):
            w, b = divmod(i, 64)
            # Check if bit i is set
            if w < W and w < len(bits) and (bits[w] >> b) & 1:
                 set_indices.append(i)

        # Calculate defenders
        for i in set_indices:
             # Check if index i is valid for out_bits
             if 0 <= i < m:
                 # defenders = arr_or_global(defenders, out_bits[i]) # Use global helper
                 out_i = out_bits[i]
                 if len(out_i) == W: # Check row length again
                    for k in range(W): defenders[k] |= out_i[k]

        # Precompute NOT defenders using global helper
        not_defenders = bitwise_not_global(defenders, W)

        # Calculate conflicts and undefended
        for i in set_indices:
             # Check if index i is valid for out_bits and in_bits
            if 0 <= i < m:
                out_i = out_bits[i]
                in_i = in_bits[i]
                # Check row lengths again for safety
                if len(out_i) == W and len(in_i) == W:
                    for k in range(W):
                        # Conflicts = popcount(out_i & bits)
                        conflicts += popcount_u64(out_i[k] & bits[k])
                        # Undefended = popcount(in_i & ~defenders)
                        undef += popcount_u64(in_i[k] & not_defenders[k])

        return conflicts + undef


# ─────────────────────────────────────────────────────────────────────────────
# Helpers to build Bit-Set arrays (NumPy or List based)
# ─────────────────────────────────────────────────────────────────────────────
def build_bitsets(af: AF, nodes: List[int]) -> tuple[Dict[int, int], Any, Any]:
    """
    Builds bitset representations (out_bits, in_bits) for the subgraph induced by 'nodes'.
    Returns NumPy arrays if NumPy is available, otherwise lists of integers.
    Output: (loc_map, out_bits, in_bits)
        loc_map: Maps original AF index -> local index (0..m-1) in the bitsets.
        out_bits/in_bits: Adjacency info in bit vector format.
    """
    m = len(nodes)
    if m == 0: # Handle empty node list
        empty_arr_np = np.zeros((0, 0), dtype=np.uint64)
        empty_arr_list : List[List[int]] = []
        empty_arr = empty_arr_np if HAVE_NUMPY else empty_arr_list
        return {}, empty_arr, empty_arr

    loc = {g: i for i, g in enumerate(nodes)} # Map global node index (in af) to local index (0 to m-1)
    W = (m + 63) // 64 # Calculate number of 64-bit words needed

    # Initialize based on NumPy availability
    if HAVE_NUMPY:
        out_bits = np.zeros((m, W), dtype=np.uint64)
        in_bits  = np.zeros((m, W), dtype=np.uint64)
        # Define set_bit_func for NumPy (though direct modification is used below)
        # set_bit_func = lambda arr, r, c_w, c_b: arr[r, c_w] | (np.uint64(1) << np.uint64(c_b))
    else:
        out_bits = [[0] * W for _ in range(m)]
        in_bits  = [[0] * W for _ in range(m)]
        # Define set_bit_func for List (though direct modification is used below)
        # set_bit_func = lambda arr, r, c_w, c_b: arr[r][c_w] | (1 << c_b)

    logging.debug(f"Building bitsets for {m} nodes, using {W} words ({'NumPy' if HAVE_NUMPY else 'List'} mode).")

    for i, g in enumerate(nodes): # i is local index, g is global index in original AF
        if not (0 <= g < af.n):
            logging.warning(f"Node index {g} from cone list is out of bounds for original AF size {af.n}. Skipping its edges.")
            continue

        # Outgoing edges from g
        for v_global in af.out_edges[g]:
            if v_global in loc: # Check if the target node is within the cone/nodes
                v_local = loc[v_global]
                w, b = divmod(v_local, 64)
                if w < W: # Ensure word index is within bounds
                    if HAVE_NUMPY: out_bits[i, w] |= (np.uint64(1) << np.uint64(b))
                    else: out_bits[i][w] |= (1 << b)

        # Incoming edges to g
        for u_global in af.in_edges[g]:
            if u_global in loc: # Check if the source node is within the cone/nodes
                u_local = loc[u_global]
                w, b = divmod(u_local, 64)
                if w < W: # Ensure word index is within bounds
                    if HAVE_NUMPY: in_bits[i, w] |= (np.uint64(1) << np.uint64(u_local % 64)) # Use u_local % 64 for clarity
                    else: in_bits[i][w] |= (1 << (u_local % 64))


    return loc, out_bits, in_bits


# ─────────────────────────────────────────────────────────────────────────────
# Backtracking Search Implementation (Recursive Version)
# ─────────────────────────────────────────────────────────────────────────────
def backtracking_solver(
    root_bits: Any, # np.ndarray | List[int]
    q_loc: int,      # Local index of the query argument in the cone
    cand: List[int], # All candidate local indices in the cone
    out_bits: Any,   # np.ndarray | List[List[int]]
    in_bits: Any,    # np.ndarray | List[List[int]]
    cap: float,      # Time cap in seconds
    progress: Optional[float] = None, # Progress reporting interval
) -> bool:
    """
    Attempts to find an admissible set containing q_loc using recursive backtracking DFS.
    Uses Numba conflict check if available. Modifies state in-place during search.
    Relies on globally defined helper functions (e.g., *_global) based on HAVE_NUMPY.
    Returns True if an admissible set containing q_loc is found, False otherwise.
    """
    start_time = time.monotonic()
    _time = time.monotonic # Local alias for performance
    logging.info(f"Starting Recursive Backtracking Search with time cap={cap:.1f}s.")

    # --- Determine dimensions m (nodes in cone) and W (bit vector width) ---
    m = 0; W = 0; is_numpy_mode = False
    if HAVE_NUMPY and isinstance(root_bits, np.ndarray) and isinstance(out_bits, np.ndarray):
        try:
            m = out_bits.shape[0] # Get m from out_bits shape
            W = root_bits.shape[0] # Get W from root_bits shape
            # Consistency check (allow m=0 case where W might be 0)
            if m > 0 and W != (m + 63) // 64:
                 # W might be different if root_bits was created differently, but should match out_bits' inferred W
                 inferred_W = (m+63)//64
                 if W != inferred_W:
                      logging.warning(f"NumPy mode: W mismatch. root_bits W={W}, inferred W={inferred_W} from m={m}. Using root_bits W.")
                 # Also check second dimension of out_bits/in_bits
                 if out_bits.shape[1] != W or (in_bits.shape[0] == m and in_bits.shape[1] != W):
                      logging.error(f"NumPy mode: Dimension mismatch. root W={W}, out shape={out_bits.shape}, in shape={in_bits.shape}")
                      return False # Error

            is_numpy_mode = True
            logging.debug(f"Backtracking running in NumPy mode (m={m}, W={W}).")
        except IndexError: # Handle empty arrays
             if root_bits.size == 0 and out_bits.size == 0: m=0; W=0; is_numpy_mode=True
             else: logging.error("Inconsistent NumPy array shapes in backtracking."); return False # Error
    elif not HAVE_NUMPY and isinstance(root_bits, list) and isinstance(out_bits, list):
        try:
            m = len(out_bits) # Get m from out_bits
            W = len(root_bits) # Get W from root_bits
            # Consistency check
            if m > 0:
                 inferred_W = (m+63)//64
                 if W == 0 : W = inferred_W # Infer W if root_bits was empty list
                 elif W != inferred_W:
                      logging.warning(f"List mode: W mismatch. root_bits W={W}, inferred W={inferred_W} from m={m}. Using root_bits W.")
                 # Check dimensions of nested lists
                 if not all(isinstance(r, list) and len(r)==W for r in out_bits) or \
                    not isinstance(in_bits, list) or len(in_bits) != m or \
                    not all(isinstance(r, list) and len(r)==W for r in in_bits):
                      logging.error(f"List mode: Dimension mismatch. root W={W}, m={m}. Check out_bits/in_bits structure.")
                      return False # Error
            elif m==0 and W==0: pass # Empty case is fine
            elif m==0 and W > 0: logging.warning(f"List mode: m=0 but W={W}>0."); # Proceed? W=0 is safer. W=0
            is_numpy_mode = False
            logging.debug(f"Backtracking running in List mode (m={m}, W={W}).")
        except TypeError: # Handle cases where inputs are not lists
             logging.error("Inconsistent List types in backtracking."); return False # Error
    else:
        # Fallback or error if types are inconsistent
        logging.error("Inconsistent types or missing NumPy for backtracking solver.")
        return False # Cannot proceed

    if m == 0:
        logging.warning("backtracking_solver: Cone is empty (m=0). Cannot find admissible set.")
        # If query was q_loc=0 in an m=0 cone, is it IN or OUT? Defined as OUT.
        return False
    if not (0 <= q_loc < m):
        logging.error(f"backtracking_solver: Invalid query local index q_loc={q_loc} for m={m}.")
        return False
    # If W is 0 but m>0, something went wrong in setup.
    if W == 0 and m > 0:
         logging.error(f"backtracking_solver: Bit vector width W=0 but cone size m={m}. Cannot proceed.")
         return False

    # Visited states check is crucial for performance
    visited_states: Set[Tuple] = set()

    # --- Initial State Checks ---
    try:
         initial_ad = ad_defect_vec(root_bits, out_bits, in_bits)
    except Exception as e:
        logging.error(f"Error calculating initial defect: {e}")
        return False # Cannot proceed if defect calculation fails

    # Use global helper, pass W
    is_q_in_root = bool(get_bit_global(root_bits, q_loc, W))
    if not is_q_in_root:
        # This indicates an issue in the calling `decide` function's setup
        logging.error("Initial state for backtracking must contain the query argument. Query not found in root_bits.")
        # Attempt to add query to root_bits if possible? Or just fail? Fail is safer.
        return False
    if initial_ad == 0:
        logging.info("Initial state (root_bits) is already admissible.")
        return True

    # --- Recursive Helper Function (Defined Inside) ---
    solution_found_flag = [False] # Use list for mutable flag accessible by nested function
    processed_count = [0]
    next_snap_time = [start_time + progress if progress and progress > 0 else float("inf")]

    # Pre-calculate constant values needed inside recursion
    m_const = m
    W_const = W
    q_loc_const = q_loc
    cap_const = cap
    start_time_const = start_time
    is_numpy_mode_const = is_numpy_mode
    set_inplace_func = set_bit_inplace_np if is_numpy_mode_const else set_bit_inplace_list
    restore_inplace_func = restore_bit_inplace_np if is_numpy_mode_const else restore_bit_inplace_list
    out_bits_const = out_bits
    in_bits_const = in_bits

    def _find_recursive(current_bits: Any, current_candidates: set[int]):
        """Recursive helper for backtracking search. Uses constants from outer scope."""
        nonlocal visited_states, solution_found_flag, processed_count, next_snap_time # Modifiable state

        # --- Base Cases & Pruning ---
        if solution_found_flag[0]: return # Stop if solution already found elsewhere
        if _time() - start_time_const >= cap_const: return # Check time limit

        processed_count[0] += 1
        # Use global helper for state hashing
        state_tuple = bits_to_tuple_global(current_bits)
        if state_tuple in visited_states: return # Already explored this state
        visited_states.add(state_tuple)

        # --- Progress Logging ---
        current_time = _time()
        if progress and current_time >= next_snap_time[0]: # Check progress > 0
            logging.info(
                f"BKTrack t={current_time - start_time_const:.1f}s | st={processed_count[0]} | vst={len(visited_states)}"
            )
            # Schedule next snapshot relative to now to avoid burst logging
            next_snap_time[0] = current_time + progress

        # --- Heuristic Candidate Ordering ---
        # Goal: Prioritize adding candidates that defend against current undefended attacks.
        external_attackers_indices = set() # Indices of nodes attacking the current set but not defended
        # 1. Find nodes currently IN the set
        # Use global helper, pass W
        current_set_indices = {i for i in range(m_const) if get_bit_global(current_bits, i, W_const)}

        if current_set_indices: # Only calculate if the set is not empty
             # 2. Calculate nodes defended by the current set
             if is_numpy_mode_const: defenders_vec = np.zeros(W_const, dtype=np.uint64)
             else: defenders_vec = [0] * W_const
             for i_set in current_set_indices:
                  # Check index bounds before accessing out_bits_const
                  if 0 <= i_set < m_const:
                      # Use global arr_or_global
                      if is_numpy_mode_const: defenders_vec |= out_bits_const[i_set]
                      else: defenders_vec = arr_or_global(defenders_vec, out_bits_const[i_set]) # List op

             # 3. Find undefended attackers for each node in the set
             not_defenders_vec = bitwise_not_global(defenders_vec, W_const) # Precompute NOT defenders
             for i_set in current_set_indices:
                 # Check index bounds before accessing in_bits_const
                 if 0 <= i_set < m_const:
                     attackers_of_i = in_bits_const[i_set]
                     # Use global arr_and_global
                     undefending_attackers_vec = arr_and_global(attackers_of_i, not_defenders_vec)

                     # Use global vec_is_zero_global
                     if not vec_is_zero_global(undefending_attackers_vec):
                         # Find indices of these undefended attackers
                         for c in range(m_const): # Check all potential attackers
                             # Use global get_bit_global, pass W
                             if get_bit_global(undefending_attackers_vec, c, W_const):
                                 external_attackers_indices.add(c)

        # 4. Prioritize candidates
        defending_candidates = []
        other_candidates = []
        if external_attackers_indices: # Only prioritize if defense is needed
            for a in current_candidates:
                 # Check index bounds for a
                 if 0 <= a < m_const:
                     attacks_by_a = out_bits_const[a]
                     is_defender = False
                     for c_attacker in external_attackers_indices:
                         # Use global helper, pass W
                         if get_bit_global(attacks_by_a, c_attacker, W_const):
                             is_defender = True; break
                     if is_defender: defending_candidates.append(a)
                     else: other_candidates.append(a)
        else: # No defense needed currently, all are 'other'
             other_candidates = list(current_candidates)

        # Create prioritized list (defenders first, shuffled within groups for randomness)
        random.shuffle(defending_candidates); random.shuffle(other_candidates)
        prioritized_candidates = defending_candidates + other_candidates
        # --- End Heuristic Ordering ---

        # --- Try adding candidates recursively ---
        for a in prioritized_candidates: # Iterate through prioritized list
            # Check flags at start of loop iteration
            if solution_found_flag[0] or _time() - start_time_const >= cap_const: return

            # Check index bounds for a before proceeding
            if not (0 <= a < m_const): continue # Skip invalid candidate index

            # --- Pruning 1: Check Conflict before adding ---
            # Use globally defined conflict checker (Numba or Python)
            out_a_row = out_bits_const[a]; in_a_row = in_bits_const[a]
            # Ensure rows are valid before passing to checker
            if (is_numpy_mode_const and (not isinstance(out_a_row, np.ndarray) or not isinstance(in_a_row, np.ndarray))) or \
               (not is_numpy_mode_const and (not isinstance(out_a_row, list) or not isinstance(in_a_row, list))):
                 logging.error(f"Invalid row type for candidate {a} in conflict check.")
                 continue # Skip candidate if data is corrupt

            if check_add_conflict_numba(a, current_bits, out_a_row, in_a_row):
                continue # Conflict, skip this candidate 'a'

            # --- Modify state IN PLACE and Recurse ---
            # Ensure index 'a' is valid for bit vector width (should be by definition if checks pass)
            # Use pre-calculated inplace functions
            set_inplace_func(current_bits, a, W_const) # Modify in place

            # Check goal state *after* modification
            try:
                ad = ad_defect_vec(current_bits, out_bits_const, in_bits_const)
            except Exception as e:
                 logging.error(f"Error calculating defect after adding {a}: {e}")
                 # Restore state before potentially erroring out or continuing
                 restore_inplace_func(current_bits, a, W_const)
                 continue # Skip this path if defect calc fails

            if ad == 0: # Found an admissible set containing the query
                logging.info(f"Recursive backtracking found AD=0 after {processed_count[0]} states.")
                solution_found_flag[0] = True
                # Restore state before returning up the recursion! Crucial!
                restore_inplace_func(current_bits, a, W_const)
                return # Found solution

            # Recurse only if goal not met AND time allows
            if _time() - start_time_const < cap_const:
                 # Pass modified bits, remove candidate 'a' from the set for next level
                _find_recursive(current_bits, current_candidates - {a})

            # Backtrack: Restore original state *after* recursive call returns
            # This happens regardless of whether the recursive call found a solution or timed out
            restore_inplace_func(current_bits, a, W_const)

            # If solution was found deeper in recursion, stop exploring siblings
            if solution_found_flag[0]: return

    # --- Initial Call to the recursive function ---
    # Use global bits_copy_global for the initial mutable copy
    mutable_root_bits = bits_copy_global(root_bits)
    # Initial candidates are all nodes in cone EXCEPT those already in root_bits
    initial_candidates_set = set(cand) - {i for i in range(m) if get_bit_global(mutable_root_bits, i, W)}

    # Handle recursion depth (Python has limits)
    default_limit = sys.getrecursionlimit()
    # Set limit based on m, ensure it's not excessively large
    required_limit = min(max(default_limit, m + 500), 3000) # Heuristic limits
    try:
        # Only set if needed and within reasonable bounds
        if required_limit > default_limit:
             sys.setrecursionlimit(required_limit)
             logging.debug(f"Temporarily set recursion depth to {required_limit} (default was {default_limit})")
    except Exception as e:
        logging.warning(f"Could not set recursion depth: {e}. Using default {default_limit}.")
        required_limit = default_limit # Use default if setting failed

    try:
        # Start recursion
        _find_recursive(mutable_root_bits, initial_candidates_set)
    except RecursionError:
         logging.error(f"!!! Python Recursion Depth Limit ({required_limit}) exceeded during backtracking !!!")
         # Restore limit before returning False
         sys.setrecursionlimit(default_limit)
         return False # Indicate failure due to recursion depth
    except Exception as e:
         logging.error(f"!!! Unexpected Error during backtracking recursion: {e} !!!", exc_info=True)
         sys.setrecursionlimit(default_limit)
         return False # Indicate failure
    finally:
         # Always restore the original recursion limit if it was changed
         if required_limit != default_limit:
             sys.setrecursionlimit(default_limit)

    # --- Return Result based on flag ---
    if solution_found_flag[0]:
        logging.info(f"Recursive Backtracking finished: FOUND Admissible Set after {processed_count[0]} states.")
        return True
    else:
        # Check if timed out or fully explored
        if _time() - start_time >= cap:
            logging.warning(f"Recursive Backtracking finished: TIMEOUT ({cap:.1f}s) after processing {processed_count[0]} states.")
        else:
            logging.info(f"Recursive Backtracking finished: EXPLORED ALL ({processed_count[0]} states). No admissible set found.")
        return False


# ─────────────────────────────────────────────────────────────────────────────
# Main Decision Function (Orchestrator)
# ─────────────────────────────────────────────────────────────────────────────
def decide(af: AF, q_ext: int, k: int, cap: float,
           task: str, # Added task parameter
           grounded: bool, # Renamed from grounded_seed for clarity
           tau: float,
           progress: Optional[float]) -> str:
    """
    Top-level decision function: orchestrates parsing, preprocessing, and solver
    based on the specified task.
    Determines if the external query argument `q_ext` is accepted for the given task.
    Returns "IN" or "OUT".
    """
    start_decision_time = time.monotonic()
    logging.info(f"--- Starting decision process for task: {task} ---")

    # ---------- 0. Map external query ID to internal ID ----------
    if af.orig2int is None or af.int2orig is None:
        logging.error("AF object is missing ID mapping information.")
        return STATUS_OUT # Cannot proceed without mapping

    if q_ext not in af.orig2int:
        logging.error(f"Query argument {q_ext} not found in the parsed AF's interacting arguments.")
        # An argument not involved in attacks cannot be defended if attacked,
        # and cannot be in grounded unless it has 0 attackers (which parse_af might handle).
        # Safest assumption for admissibility is OUT if not in the map.
        return STATUS_OUT

    q_internal_orig = af.orig2int[q_ext] # Internal index in the *original* parsed AF
    logging.info(f"Query: external {q_ext} mapped to original internal {q_internal_orig}")

    # ---------- 1. Grounded Extension Check (Always First) ----------
    logging.info("Step 1: Calculating Grounded Extension...")
    try:
        G = grounded_extension(af)
    except Exception as e:
         logging.error(f"Failed to compute grounded extension: {e}. Returning OUT.")
         return STATUS_OUT

    if q_internal_orig in G:
        logging.info(f"Query {q_ext} (internal {q_internal_orig}) is IN the Grounded Extension.")
        logging.info(f"--- Decision Result ({task}): {STATUS_IN} (from Grounded) ---")
        return STATUS_IN
    else:
        logging.info(f"Query {q_ext} not in Grounded Extension (size {len(G)}).")

    # ---------- 2. Task-Specific Logic ----------
    # Query is NOT in the Grounded Extension, proceed based on task.

    # --- Task Group 1: DS* (except DS-ST) and DC-ID ---
    # These only rely on the grounded check result. Since q is not in G, the answer is OUT.
    if (task.startswith("DS") and task != "DS-ST") or task == "DC-ID":
        logging.info(f"Task {task} relies only on Grounded. Query not in Grounded.")
        logging.info(f"--- Decision Result ({task}): {STATUS_OUT} ---")
        return STATUS_OUT

    # --- Task Group 2: DS-ST ---
    # Use the SCC Reinstatement Heuristic.
    elif task == "DS-ST":
        logging.info("Step 2 (DS-ST): Running SCC Reinstatement Heuristic...")
        try:
            scc_result = run_scc_reinstatement(af, q_internal_orig)
            logging.info(f"SCC Reinstatement Heuristic result: {scc_result}")
            logging.info(f"--- Decision Result ({task}): {scc_result} ---")
            return scc_result
        except Exception as e:
            logging.error(f"Error during SCC Reinstatement Heuristic: {e}. Returning OUT.")
            return STATUS_OUT

    # --- Task Group 3: DC* (except DC-ID) ---
    # Use the original Backtracking approach with preprocessing.
    elif task.startswith("DC") and task != "DC-ID":
        logging.info(f"Step 2 (DC Task {task}): Proceeding with Backtracking approach.")

        # ---------- 2a. SCC Condensation and Reduction ----------
        logging.info("Performing SCC decomposition for potential reduction...")
        try:
            sccs = strongly_connected_components(af)
        except Exception as e:
             logging.error(f"Error during SCC decomposition: {e}. Proceeding without reduction.")
             sccs = [] # Treat as no reduction possible

        af_current = af               # The AF we'll work on (might be subgraph)
        q_current = q_internal_orig # Query index in af_current
        G_current = G               # Grounded set mapped to af_current indices

        if not sccs: # Handle empty graph or potential SCC error/no components
            if af.n > 0:
                logging.warning("SCC decomposition resulted in no components or failed. Proceeding with original graph.")
            # Keep af_current = af, q_current = q_internal_orig, G_current = G
        else:
            # Map each node to its SCC index
            # Use numpy array for efficiency if available
            node_to_scc_index: Any = np.full(af.n, -1, dtype=int) if HAVE_NUMPY else [-1] * af.n
            valid_sccs = True
            for i, comp in enumerate(sccs):
                for node in comp:
                    if 0 <= node < af.n: node_to_scc_index[node] = i
                    else:
                        logging.warning(f"Node index {node} from SCC component {i} out of bounds for AF size {af.n}. SCC map invalid.")
                        valid_sccs = False; break
                if not valid_sccs: break

            # Find the SCC containing the query
            if not valid_sccs or not (0 <= q_internal_orig < af.n):
                 logging.error(f"Query index {q_internal_orig} out of bounds or SCC map invalid. Proceeding without SCC reduction.")
                 # Keep af_current = af etc.
            else:
                q_scc_index = node_to_scc_index[q_internal_orig]

                if q_scc_index == -1:
                    logging.error(f"Query node {q_internal_orig} was not assigned to any SCC component. Check SCC logic. Proceeding without SCC reduction.")
                    # Keep af_current = af etc.
                else:
                    logging.debug(f"Query node {q_internal_orig} belongs to SCC index {q_scc_index}.")
                    # Build condensation graph (DAG) to find ancestor SCCs
                    num_sccs = len(sccs)
                    dag_parents = [set() for _ in range(num_sccs)] # Stores parent SCC indices for each SCC

                    # Populate dag_parents by iterating through original edges
                    for u_orig in range(af.n):
                        u_scc = node_to_scc_index[u_orig]
                        if u_scc == -1: continue # Skip nodes not in a valid SCC
                        for v_orig in af.out_edges[u_orig]:
                             # Check bounds for v_orig and its mapping
                            if 0 <= v_orig < af.n:
                                v_scc = node_to_scc_index[v_orig]
                                # Add edge in DAG if nodes are in different valid SCCs
                                if v_scc != -1 and u_scc != v_scc:
                                    dag_parents[v_scc].add(u_scc)

                    # Collect ancestor SCCs using BFS/DFS starting from query's SCC
                    sccs_to_keep_indices: Set[int] = set()
                    queue: List[int] = [q_scc_index] # Use list as queue
                    processed: Set[int] = {q_scc_index} # Mark start node as processed
                    sccs_to_keep_indices.add(q_scc_index)

                    head = 0
                    while head < len(queue):
                         s_idx = queue[head]
                         head += 1
                         # Add valid, unprocessed parents to the queue
                         if 0 <= s_idx < num_sccs: # Check index validity
                             for parent_idx in dag_parents[s_idx]:
                                 if parent_idx not in processed:
                                      processed.add(parent_idx)
                                      sccs_to_keep_indices.add(parent_idx)
                                      queue.append(parent_idx)


                    # --- Subgraph construction if reduction possible ---
                    if len(sccs_to_keep_indices) < num_sccs:
                        logging.info(f"SCC reduction: Keeping {len(sccs_to_keep_indices)} out of {num_sccs} SCCs (ancestors of query SCC).")
                        # Gather all original node indices from the kept SCCs
                        kept_nodes_orig_indices = sorted([
                            node for s_idx in sccs_to_keep_indices
                             if 0 <= s_idx < len(sccs) # Safety check on s_idx
                             for node in sccs[s_idx] if 0 <= node < af.n # Double check node index validity
                        ])

                        if not kept_nodes_orig_indices:
                            logging.error("SCC reduction resulted in zero nodes to keep (query SCC was empty or invalid?). Returning OUT.")
                            return STATUS_OUT
                        if q_internal_orig not in kept_nodes_orig_indices:
                            logging.error(f"Query node {q_internal_orig} was lost during SCC reduction process. Returning OUT.")
                            return STATUS_OUT

                        n_subgraph = len(kept_nodes_orig_indices)
                        subgraph_remap = {orig_idx: new_idx for new_idx, orig_idx in enumerate(kept_nodes_orig_indices)}

                        # Create the new AF subgraph object
                        sub_af = AF(n_subgraph)
                        # Create mappings for the subgraph (optional but good practice)
                        # Map original external IDs if possible
                        if af.int2orig and af.orig2int:
                            try:
                                sub_af.int2orig = [af.int2orig[orig_idx] for orig_idx in kept_nodes_orig_indices]
                                sub_af.orig2int = {orig_ext_id: new_idx for new_idx, orig_ext_id in enumerate(sub_af.int2orig)}
                            except (IndexError, TypeError, KeyError):
                                 logging.warning("Could not create full original ID mapping for subgraph.")
                                 sub_af.int2orig = None # Mark as unavailable
                                 sub_af.orig2int = None
                        else: # Original mapping was missing
                             sub_af.int2orig = None
                             sub_af.orig2int = None


                        # Add edges to the subgraph
                        num_sub_edges = 0
                        for u_orig in kept_nodes_orig_indices:
                            u_new = subgraph_remap[u_orig]
                            for v_orig in af.out_edges[u_orig]:
                                if v_orig in subgraph_remap: # Check if target is also kept
                                    v_new = subgraph_remap[v_orig]
                                    sub_af.add_edge(u_new, v_new)
                                    num_sub_edges += 1

                        logging.info(f"Subgraph created with {sub_af.n} nodes and {num_sub_edges} edges.")
                        af_current = sub_af
                        q_current = subgraph_remap[q_internal_orig]
                        G_current = {subgraph_remap[g] for g in G if g in subgraph_remap} # Remap grounded set too

                    else: # No reduction occurred
                        logging.info("SCC analysis did not result in graph reduction.")
                        # Keep af_current = af, q_current = q_internal_orig, G_current = G

        # ------------------------------------------------------------------
        # 2b. Local Cone Extraction on af_current (after potential reduction)
        # ------------------------------------------------------------------
        if not (0 <= q_current < af_current.n):
             logging.error(f"Query index {q_current} is out of bounds for current AF size {af_current.n} before cone extraction.")
             return STATUS_OUT

        logging.info(f"Extracting {k}-hop cone around query node {q_current} in the current graph ({af_current.n} nodes)...")
        try:
            cone_nodes_indices_set = k_hop_cone(af_current, q_current, k)
        except Exception as e:
             logging.error(f"Error during k-hop cone extraction: {e}. Returning OUT.")
             return STATUS_OUT


        if not cone_nodes_indices_set:
             # If the cone is empty but q_current was valid, it implies q_current is isolated?
             # k_hop_cone should always return at least {q} if q is valid.
             logging.warning(f"k-hop cone around query {q_current} is unexpectedly empty (but node exists). Result is OUT.")
             return STATUS_OUT
        # Ensure query is included (should be by k_hop_cone definition)
        if q_current not in cone_nodes_indices_set:
             logging.warning(f"Query node {q_current} not included in its own {k}-hop cone? Adding it.")
             cone_nodes_indices_set.add(q_current)

        # Convert to sorted list for deterministic bitset building
        cone_nodes_indices = sorted(list(cone_nodes_indices_set))
        m_cone = len(cone_nodes_indices)
        logging.info(f"Cone size: {m_cone} nodes.")

        # ------------------------------------------------------------------
        # 2c. Prepare for Solver: Build Bitsets for the Cone
        # ------------------------------------------------------------------
        # cone_loc_map: maps node index in af_current -> local index in cone (0 to m_cone-1)
        # outb, inb: bitset adjacency matrices (NumPy or List based) for the cone
        try:
            cone_loc_map, outb, inb = build_bitsets(af_current, cone_nodes_indices)
        except Exception as e:
            logging.error(f"Error building bitsets for the cone: {e}. Returning OUT.")
            return STATUS_OUT


        if q_current not in cone_loc_map:
            logging.error(f"Query node {q_current} (in af_current) not found in the cone's local map after building bitsets. Cannot run solver.")
            return STATUS_OUT

        q_cone_local = cone_loc_map[q_current] # Local index of query within the cone bitsets
        W_cone = 0 # Width of the bit vectors
        try: # Determine W safely
            if HAVE_NUMPY and isinstance(outb, np.ndarray):
                W_cone = outb.shape[1] if len(outb.shape) > 1 else 0 # Handle 0-node case (shape=(0,))
            elif isinstance(outb, list): # Check list case
                 # Handle empty list or list of empty lists
                W_cone = len(outb[0]) if m_cone > 0 and outb and isinstance(outb[0], list) else 0
            elif m_cone == 0: W_cone = 0 # Empty cone
            # Ensure W is consistent if m > 0
            if m_cone > 0 and W_cone == 0: W_cone = (m_cone + 63) // 64
        except Exception as e:
            logging.error(f"Could not determine bitset width W_cone: {e}. Returning OUT.")
            return STATUS_OUT


        logging.debug(f"Cone bitsets created: m={m_cone}, W={W_cone}. Query local index: {q_cone_local}.")

        # ------------------------------------------------------------------
        # 2d. Initialize Solver Root State (Bit Vector)
        # ------------------------------------------------------------------
        if m_cone == 0: # Should have been caught by cone check, but safety first
            logging.info("Cone is empty, result is OUT.")
            return STATUS_OUT
        if W_cone == 0 and m_cone > 0: # Consistency check
            logging.error(f"Inconsistent cone bitset dimensions: W=0 but m={m_cone}>0.")
            return STATUS_OUT

        # Initialize root state: a bit vector representing a subset of cone nodes.
        # Start with only the query node included.
        if HAVE_NUMPY:
            root_bits = np.zeros(W_cone, dtype=np.uint64)
        else:
            root_bits = [0] * W_cone

        # Set the bit corresponding to the local query index
        w_q_local, b_q_local = divmod(q_cone_local, 64)
        if w_q_local < W_cone:
            if HAVE_NUMPY: root_bits[w_q_local] |= (np.uint64(1) << np.uint64(b_q_local))
            else: root_bits[w_q_local] |= (1 << b_q_local)
        else:
            logging.error(f"Query local index {q_cone_local} resulted in word index {w_q_local} out of bounds for W={W_cone}.")
            return STATUS_OUT

        # Optional: Seed root_bits with Grounded Extension elements within the cone
        num_grounded_in_cone = 0
        if grounded and G_current: # Only seed if grounded was computed and non-empty
            # Find grounded nodes (indices in af_current) that are *also* in the cone (via cone_loc_map)
            grounded_in_cone_local = [cone_loc_map[g] for g in G_current if g in cone_loc_map]
            num_grounded_in_cone = len(grounded_in_cone_local)

            # Seed only if grounded set within cone is non-empty and relatively small
            use_grounded_seed = (num_grounded_in_cone > 0 and num_grounded_in_cone <= tau * m_cone)

            logging.info(f"Grounded seed check: {num_grounded_in_cone}/{m_cone} grounded nodes in cone. tau={tau}. Seeding: {use_grounded_seed}.")
            if use_grounded_seed:
                for g_local_idx in grounded_in_cone_local:
                    w_g, b_g = divmod(g_local_idx, 64)
                    if w_g < W_cone:
                        if HAVE_NUMPY: root_bits[w_g] |= (np.uint64(1) << np.uint64(b_g))
                        else: root_bits[w_g] |= (1 << b_g)
                # Recalculate defect of seeded root state for logging/debugging if needed
                try:
                     initial_ad_seeded = ad_defect_vec(root_bits, outb, inb)
                     logging.debug(f"Root state seeded with grounded. Initial defect: {initial_ad_seeded}")
                except Exception: pass # Ignore defect calc error here
            else:
                logging.info("Grounded seed not used (too large, empty, or disabled).")


        # ------------------------------------------------------------------
        # 2e. Prepare Solver Inputs
        # ------------------------------------------------------------------
        # Candidates for the solver are all nodes within the cone (local indices)
        cand_local_indices = list(range(m_cone))

        # Calculate remaining time budget for the solver
        elapsed_preprocessing = time.monotonic() - start_decision_time
        # Ensure at least a small positive time budget
        solver_time_budget = max(0.1, cap - elapsed_preprocessing)
        logging.info(f"Preprocessing time: {elapsed_preprocessing:.2f}s. Solver budget: {solver_time_budget:.2f}s.")

        # ------------------------------------------------------------------
        # 2f. Run the Backtracking Solver
        # ------------------------------------------------------------------
        logging.info(f"Starting Backtracking Search on cone: time={solver_time_budget:.1f}s, progress={progress}")
        try:
            found_admissible_set = backtracking_solver(
                root_bits=root_bits,
                q_loc=q_cone_local,
                cand=cand_local_indices, # Pass all local indices as candidates
                out_bits=outb,
                in_bits=inb,
                cap=solver_time_budget,
                progress=progress
            )
        except Exception as e:
             logging.error(f"Critical error during backtracking_solver execution: {e}. Returning OUT.", exc_info=True)
             return STATUS_OUT


        # ------------------------------------------------------------------
        # 2g. Return Final Result for DC task
        # ------------------------------------------------------------------
        result = STATUS_IN if found_admissible_set else STATUS_OUT
        total_decision_time = time.monotonic() - start_decision_time
        logging.info(f"Solver finished. Final Decision: {result}. Total time: {total_decision_time:.2f}s.")
        logging.info(f"--- Decision Result ({task}): {result} ---")
        return result

    # --- Fallback for Unknown Task ---
    else:
        logging.warning(f"Unknown task specified: '{task}'. Grounded check failed. Defaulting to OUT.")
        logging.info(f"--- Decision Result ({task}): {STATUS_OUT} (Unknown Task) ---")
        return STATUS_OUT


# ─────────────────────────────────────────────────────────────────────────────
# Command‑line interface
# ─────────────────────────────────────────────────────────────────────────────
@hard_timeout(580) # Apply overall wall-clock timeout (slightly less than typical competition limits)
def _run():
    if HAVE_NUMBA == False or HAVE_NUMPY == False:
        print("NUMBA and NUMPY are required for the ICCMA version.")
        exit()

    p = argparse.ArgumentParser(
        description="Task-based solver for Admissibility in Argumentation Frameworks.",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
        )
    p.add_argument("file", help="Path to the AF file (e.g., .apx format)")
    p.add_argument("query", type=int, help="The query argument ID (external ID from file)")
    p.add_argument("--task", type=str, default="DC-CO",
                   help="Task to solve (e.g., DS-PR, DC-CO, DS-ST, DC-ID). Determines the algorithm used.")
    p.add_argument("--time", type=float, default=55.0, help="Wall-clock time limit in seconds for the decision process")
    # Args specific to Backtracking (DC* tasks)
    p.add_argument("--k", type=int, default=3, help="Radius for the k-hop cone extraction (used for DC* tasks)")
    p.add_argument("--grounded-seed", dest="grounded", action="store_true", default=True, help="Enable seeding solver start state with grounded extension (DC* tasks, if small enough)")
    p.add_argument("--no-grounded-seed", dest="grounded", action="store_false", help="Disable seeding with grounded extension (DC* tasks)")
    p.add_argument("--tau", type=float, default=0.1, help="Grounded seed relative size threshold (<= tau * cone_size, DC* tasks)")
    p.add_argument("--progress", type=float, default=10.0, help="Log progress every N seconds during solver (DC* tasks, 0=off)")
    # General args
    p.add_argument("--debug", action="store_true", help="Enable debug logging")
    p.add_argument("--log-level", default="CRITICAL", choices=["DEBUG", "INFO", "WARNING", "ERROR", "CRITICAL"], help="Set logging level")
    args = p.parse_args()

    setup_logging(args.debug, args.log_level)
    # Censor potentially sensitive info if needed before logging
    log_args = vars(args).copy()
    # Example: if args.file contained sensitive path info, censor it here
    # log_args['file'] = '...'
    logging.info(f"Starting solver with arguments: {log_args}")
    logging.info(f"NumPy Available: {HAVE_NUMPY}")
    logging.info(f"Numba Available: {HAVE_NUMBA}")


    try:
        af = parse_af(args.file)
        if af is None:
            logging.error("Failed to parse AF file. Exiting.")
            print(STATUS_OUT) # Consistent output on parsing failure
            sys.exit(1)

        # Call decide with the new task parameter
        result = decide(
            af=af,
            q_ext=args.query,
            k=args.k,          # Used only by DC* tasks within decide
            cap=args.time,
            task=args.task,    # Pass the task string
            grounded=args.grounded, # Used only by DC* tasks within decide
            tau=args.tau,      # Used only by DC* tasks within decide
            progress=(None if args.progress <= 0 else args.progress), # Used by DC*
        )
        print(result) # Print final result (IN/OUT) to stdout
        sys.exit(0) # Exit with success code

    # Handle specific, potentially recoverable errors if needed
    # FileNotFoundError should be caught by parse_af returning None now
    except TimeoutError as e:
        # This could be triggered by hard_timeout or potentially other timeouts within libs
        logging.warning(f"Operation timed out: {e}. Returning OUT.")
        print(STATUS_OUT)
        sys.exit(0) # Exit normally on timeout as per competition rules often
    except MemoryError:
        logging.error("Insufficient memory to complete the operation. Returning OUT.")
        print(STATUS_OUT)
        sys.exit(1) # Exit with error code for memory issues
    except Exception as e:
        # Catch any other unexpected exceptions during execution
        logging.exception(f"An unexpected critical error occurred: {e}") # Log full traceback
        print(STATUS_OUT) # Default to OUT on unexpected errors
        sys.exit(1) # Exit with error code


if __name__ == "__main__":
    # Wrap the main execution in a final try block for ultimate safety net
    try:
        _run()
    except SystemExit as e:
         # Allow sys.exit calls to propagate normally
         raise e
    except Exception as e:
        # Catch exceptions that might escape _run's handlers (e.g., argparse errors)
        # Logging might not be set up yet if error is early
        print(f"[CRITICAL] Unhandled exception at top level: {e}", file=sys.stderr)
        print(STATUS_OUT) # Default to OUT if utterly failed before/during setup
        sys.exit(1)