#!/bin/bash
# Wrapper script for the Python Admissibility Solver (solver.py)
# Adapted for ICCMA interface.

# function for echoing on standard error
function echoerr()
{
    # to remove standard error echoing, please comment the following line
    echo "$@" 1>&2;
}

################################################################
# C O N F I G U R A T I O N
#
# Customize author/version and ensure paths/python command are correct.

# output information
function information()
{
    echo "Heuback Approximate Admissibility Solver v0.1" # Updated Version slightly
    echo "Lars Malmqvist"             # Updated Author
}

# how to invoke the Python solver:
function solver()
{
    local fileinput=$1  # input file with correct path ($local_fileinput)
    local task=$2       # task to solve ($local_problem, e.g., DC-CO)
    local additional=$3 # additional information, i.e., name of the target argument ($local_additional)

    # Determine the directory where this script and solver.py reside
    local DIR
    DIR=$( cd -- "$( dirname -- "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )
    local SOLVER_SCRIPT="$DIR/heuback.py" # Assuming solver.py is in the same directory

    if [ ! -f "$SOLVER_SCRIPT" ]; then
        echoerr "ERROR: Solver script not found at $SOLVER_SCRIPT"
        # Output equivalent of "OUT" for solver failure in ICCMA context?
        # Usually, scripts should output nothing on stderr and just exit on error,
        # but returning OUT might be expected by some test harnesses on setup failure.
        # Let's echo OUT and exit non-zero.
        echo "OUT"
        exit 1
    fi

    # Construct the command line for the new solver.py
    # Takes file and query as positional args, task via --task
    # Other parameters like --time, --k use defaults within the Python script.
    echoerr "Executing: python3 \"$SOLVER_SCRIPT\" \"$fileinput\" \"$additional\" --task \"$task\""
    python3 "$SOLVER_SCRIPT" "$fileinput" "$additional" --task "$task"
    # Capture exit code if needed for debugging
    local exit_code=$?
    if [ $exit_code -ne 0 ]; then
        echoerr "WARNING: Python solver exited with status $exit_code"
        # If the python script failed badly and didn't print IN/OUT,
        # we might need to print OUT here as a fallback. However,
        # the python script has try/except blocks that should print OUT.
    fi
}


# --- Supported Formats ---
# The python script reads a specific "p af N" format. Assuming 'apx' is compatible.
formats=""
formats="${formats} apx" # Aspartix format (or compatible)

# --- Supported Tasks ---
# Specific list provided by user. The Python script handles these via prefix/exact matching.
tasks=""
tasks="${tasks} DS-PR"
tasks="${tasks} DC-CO"
tasks="${tasks} DS-ST"
tasks="${tasks} DS-SST"
tasks="${tasks} DC-ST"
tasks="${tasks} DC-SST"
tasks="${tasks} DC-ID"


function list_output()
{
    local check_something_printed=false
    printf "["
    if [[ "$1" = "1" ]]; then # List formats
        for format in ${formats}; do
            if [ "$check_something_printed" = true ]; then
                printf ", "
            fi
            printf "%s" "$format"
            check_something_printed=true
        done
    elif [[ "$1" = "2" ]]; then # List problems/tasks
        for task in ${tasks}; do
            if [ "$check_something_printed" = true ]; then
                printf ", "
            fi
            printf "%s" "$task"
            check_something_printed=true
        done
    fi
    printf "]\n"
}

# --- Main Execution Logic ---
function main()
{
    if [ "$#" = "0" ]; then
        information
        exit 0
    fi

    local local_problem=""
    local local_fileinput=""
    local local_format="apx" # Default format assumption
    local local_additional=""

    local local_task_valid=""

    while [ "$1" != "" ]; do
        case $1 in
            "--formats")
                list_output 1
                exit 0
                ;;
            "--problems")
                list_output 2
                exit 0
                ;;
            "-p")
                shift
                local_problem=$1
                ;;
            "-f")
                shift
                local_fileinput=$1
                ;;
            "-fo")
                shift
                local_format=$1 # Format argument is parsed but not explicitly used by this python script
                ;;
            "-a")
                shift
                local_additional=$1
                ;;
            *)
                echoerr "Unknown option: $1"
                # exit 1 # Optional: exit on unknown options
                ;;
        esac
        shift
    done

    # Validate task
    if [ -z "$local_problem" ]; then
        echoerr "ERROR: Task (-p) missing."
        exit 1
    else
        for local_task in ${tasks}; do
            if [ "$local_task" = "$local_problem" ]; then
                local_task_valid="true"
                break # Found a valid task
            fi
        done
        if [ -z "$local_task_valid" ]; then
            # Capture list output for error message
            local supported_tasks_list
            supported_tasks_list=$(list_output 2 | tr -d '\n')
            echoerr "ERROR: Task '$local_problem' is not supported by this solver."
            echoerr "Supported tasks are: ${supported_tasks_list}"
            exit 1
        fi
    fi

    # Validate input file
    if [ -z "$local_fileinput" ]; then
        echoerr "ERROR: Input file (-f) missing."
        exit 1
    fi
    if [ ! -f "$local_fileinput" ]; then
         echoerr "ERROR: Input file '$local_fileinput' not found."
         exit 1
    fi

    # Validate additional argument for relevant tasks (DC/DS require an argument)
    if [[ "$local_problem" == "DC-"* || "$local_problem" == "DS-"* ]]; then
        if [ -z "$local_additional" ]; then
            echoerr "ERROR: Argument (-a) required for task '$local_problem'."
            exit 1
        fi
    fi

    # Call the solver function
    # Result is echoed directly by the solver function now
    solver "$local_fileinput" "$local_problem" "$local_additional"

}

# Execute main function with all script arguments
main "$@"
exit 0