use super::{cli_manager, command::Command, common};
use anyhow::{Context, Result};
use clap::{AppSettings, Arg, ArgMatches, SubCommand};
use scalop::{
    io::{AspartixReader, Iccma23Reader, InstanceReader},
    solvers::{
        CompleteSemanticsSolver, ExtensionChecker, GroundedSemanticsSolver, IdealSemanticsSolver,
        PreferredSemanticsSolver, SemiStableSemanticsSolver, StableSemanticsSolver,
        StageSemanticsSolver,
    },
    utils::LabelType,
};

const CMD_NAME: &str = "check-extension";

const ARG_SEM: &str = "ARG_SEM";

pub(crate) struct CheckExtensionCommand;

impl CheckExtensionCommand {
    pub fn new() -> Self {
        CheckExtensionCommand
    }
}

impl<'a> Command<'a> for CheckExtensionCommand {
    fn name(&self) -> &str {
        CMD_NAME
    }

    fn clap_subcommand(&self) -> clap::App<'a, 'a> {
        SubCommand::with_name(CMD_NAME)
            .about("Checks if an extension belongs to an AF")
            .setting(AppSettings::DisableVersion)
            .arg(common::input_args())
            .arg(common::reader_arg())
            .arg(cli_manager::logging_level_cli_arg())
            .arg(
                Arg::with_name(ARG_SEM)
                    .short("s")
                    .long("semantics")
                    .empty_values(false)
                    .multiple(false)
                    .possible_values(&["GR", "CO", "PR", "ST", "SST", "STG", "ID"])
                    .help("the semantics to consider")
                    .required(true),
            )
    }

    fn execute(&self, arg_matches: &ArgMatches<'_>) -> Result<()> {
        match arg_matches.value_of(common::ARG_READER).unwrap() {
            "apx" => execute_with_reader(arg_matches, &mut AspartixReader::default()),
            "iccma23" => execute_with_reader(arg_matches, &mut Iccma23Reader::default()),
            _ => unreachable!(),
        }
    }
}

fn execute_with_reader<T>(
    arg_matches: &ArgMatches<'_>,
    reader: &mut dyn InstanceReader<T>,
) -> Result<()>
where
    T: LabelType,
{
    let file = arg_matches.value_of(common::ARG_INPUT).unwrap();
    let af = common::read_file_path(file, reader)?;
    let solver: Box<dyn ExtensionChecker<T>> = match arg_matches.value_of(ARG_SEM).unwrap() {
        "GR" => Box::new(GroundedSemanticsSolver::new(&af)),
        "CO" => Box::new(CompleteSemanticsSolver::new(&af)),
        "PR" => Box::new(PreferredSemanticsSolver::new(&af)),
        "ST" => Box::new(StableSemanticsSolver::new(&af)),
        "SST" => Box::new(SemiStableSemanticsSolver::new(&af)),
        "STG" => Box::new(StageSemanticsSolver::new(&af)),
        "ID" => Box::new(IdealSemanticsSolver::new(&af)),
        _ => unreachable!(),
    };
    let ext = reader
        .read_extension(&af, &mut std::io::stdin())
        .context("while reading a potential extension")?;
    if solver.is_extension(&ext.iter().map(|a| a.label()).collect::<Vec<_>>()) {
        println!("YES")
    } else {
        println!("NO")
    }
    Ok(())
}
